const { app, BrowserWindow, ipcMain, dialog } = require('electron');
const path = require('path');
const fs = require('fs');
const os = require('os');
const { exec } = require('child_process');
const crypto = require('crypto');
const { v4: uuidv4 } = require('uuid');
const axios = require('axios');
const https = require('https');
const http = require('http');

// 运行时保护 (已禁用，避免模块缺失错误)
// const RuntimeProtection = require('./scripts/runtime-protection');
// let protection;

// 使用新的数据库封装模块
const CursorDatabase = require('./database');

// 是否是开发环境
const isDev = process.argv.includes('--dev') || process.env.NODE_ENV === 'development';
let mainWindow;

// 设备ID监控相关变量
let deviceIdWatcher = null;
let protectedDeviceIds = null;
let storageJsonPath = null;
let isWatcherActive = false;

// 代理自动切换定时器相关变量
let proxyAutoSwitchTimer = null;
let isProxyAutoSwitchEnabled = false;
const PROXY_SWITCH_INTERVAL = 12 * 60 * 60 * 1000; // 12小时（毫秒）

// 启动设备ID保护监控
function startDeviceIdProtection(storagePath, deviceIds) {
  try {
    // 停止之前的监控
    stopDeviceIdProtection();

    storageJsonPath = storagePath;
    protectedDeviceIds = { ...deviceIds };

    console.log('🛡️ 启动设备ID保护监控...');
    console.log(`📁 监控文件: ${storageJsonPath}`);
    console.log(`🔒 保护的设备ID:`, {
      devDeviceId: protectedDeviceIds['telemetry.devDeviceId'],
      macMachineId: protectedDeviceIds['telemetry.macMachineId']?.substring(0, 12) + '...',
      machineId: protectedDeviceIds['telemetry.machineId']?.substring(0, 12) + '...',
      sqmId: protectedDeviceIds['telemetry.sqmId']
    });

    // 创建文件监控器
    deviceIdWatcher = fs.watchFile(storageJsonPath, { interval: 2000 }, (curr, prev) => {
      if (curr.mtime !== prev.mtime && isWatcherActive) {
        console.log('📝 检测到storage.json文件被修改，检查设备ID...');
        setTimeout(() => {
          restoreProtectedDeviceIds();
        }, 1000); // 延迟1秒确保文件写入完成
      }
    });

    isWatcherActive = true;
    console.log('✅ 设备ID保护监控已启动');

    return true;
  } catch (error) {
    console.error('❌ 启动设备ID保护失败:', error);
    return false;
  }
}

// 停止设备ID保护监控
function stopDeviceIdProtection() {
  try {
    if (deviceIdWatcher) {
      fs.unwatchFile(storageJsonPath);
      deviceIdWatcher = null;
    }
    isWatcherActive = false;
    console.log('🛑 设备ID保护监控已停止');
    return true;
  } catch (error) {
    console.error('❌ 停止设备ID保护失败:', error);
    return false;
  }
}

// 恢复被保护的设备ID
function restoreProtectedDeviceIds() {
  try {
    if (!storageJsonPath || !protectedDeviceIds || !fs.existsSync(storageJsonPath)) {
      return false;
    }

    const content = fs.readFileSync(storageJsonPath, 'utf8');
    const storageData = JSON.parse(content);

    let hasChanges = false;
    let changedFields = [];

    // 检查并恢复每个受保护的设备ID
    for (const [key, protectedValue] of Object.entries(protectedDeviceIds)) {
      if (storageData[key] !== protectedValue) {
        console.log(`🔧 恢复被修改的设备ID: ${key}`);
        console.log(`  原值: ${storageData[key]}`);
        console.log(`  恢复为: ${protectedValue}`);

        storageData[key] = protectedValue;
        hasChanges = true;
        changedFields.push(key);
      }
    }

    if (hasChanges) {
      // 临时停止监控，避免触发递归
      const wasActive = isWatcherActive;
      isWatcherActive = false;

      // 写入修正后的数据
      fs.writeFileSync(storageJsonPath, JSON.stringify(storageData, null, 2), 'utf8');

      console.log(`✅ 已恢复 ${changedFields.length} 个被修改的设备ID:`, changedFields);

      // 延迟重新启动监控
      setTimeout(() => {
        if (wasActive) {
          isWatcherActive = true;
          console.log('🛡️ 设备ID保护监控已重新激活');
        }
      }, 3000);

      return true;
    } else {
      console.log('✅ 设备ID未被修改，无需恢复');
      return false;
    }

  } catch (error) {
    console.error('❌ 恢复设备ID失败:', error);
    return false;
  }
}

// HTTP请求工具函数
function makeHttpRequest(url, options = {}) {
  return new Promise((resolve, reject) => {
    const urlObj = new URL(url);
    const client = urlObj.protocol === 'https:' ? https : http;

    const requestOptions = {
      hostname: urlObj.hostname,
      port: urlObj.port,
      path: urlObj.pathname + urlObj.search,
      method: options.method || 'GET',
      headers: {
        'Content-Type': 'application/json',
        ...options.headers
      },
      timeout: options.timeout || 5000 // 5秒超时
    };

    const req = client.request(requestOptions, (res) => {
      let data = '';
      res.on('data', (chunk) => {
        data += chunk;
      });
      res.on('end', () => {
        try {
          const result = JSON.parse(data);
          resolve(result);
        } catch (parseError) {
          reject(new Error(`解析响应失败: ${parseError.message}`));
        }
      });
    });

    req.on('error', (error) => {
      reject(new Error(`请求失败: ${error.message}`));
    });

    req.on('timeout', () => {
      req.destroy();
      reject(new Error('请求超时'));
    });

    if (options.data) {
      req.write(JSON.stringify(options.data));
    }

    req.end();
  });
}

// 设置自定义缓存路径
const userDataPath = app.getPath('userData');
const cachePath = path.join(userDataPath, 'Cache');
// 确保缓存目录存在
if (!fs.existsSync(cachePath)) {
  try {
    fs.mkdirSync(cachePath, { recursive: true });
  } catch (err) {
    console.error('创建缓存目录失败:', err);
  }
}
app.commandLine.appendSwitch('disk-cache-dir', cachePath);

// 性能优化设置
app.commandLine.appendSwitch('disable-gpu-sandbox');
app.commandLine.appendSwitch('disable-software-rasterizer');
app.commandLine.appendSwitch('disable-background-timer-throttling');
app.commandLine.appendSwitch('disable-backgrounding-occluded-windows');
app.commandLine.appendSwitch('disable-renderer-backgrounding');
app.commandLine.appendSwitch('disable-features', 'TranslateUI');
app.commandLine.appendSwitch('disable-ipc-flooding-protection');
app.commandLine.appendSwitch('no-sandbox');
// 启用硬件加速以提升性能（如果系统支持）
// app.disableHardwareAcceleration();

// 创建主窗口
function createWindow() {
  // 获取屏幕尺寸
  const { screen } = require('electron');
  const primaryDisplay = screen.getPrimaryDisplay();
  const { width: screenWidth, height: screenHeight } = primaryDisplay.workAreaSize;

  // 计算窗口尺寸（v3.0 UI 需要更大空间）
  const windowWidth = Math.min(1280, Math.floor(screenWidth * 0.75));
  const windowHeight = Math.min(850, Math.floor(screenHeight * 0.85));

  // 计算窗口位置（居中）
  const x = Math.floor((screenWidth - windowWidth) / 2);
  const y = Math.floor((screenHeight - windowHeight) / 2);

  mainWindow = new BrowserWindow({
    width: windowWidth,
    height: windowHeight,
    x: x,
    y: y,
    minWidth: 1200,
    minHeight: 800,
    maxWidth: windowWidth,
    maxHeight: windowHeight,
    center: true, // 确保窗口居中
    resizable: false, // 禁止调整大小
    maximizable: false, // 禁用最大化按钮
    autoHideMenuBar: true, // 隐藏菜单栏
    backgroundColor: '#203a43', // v3.0 深蓝色背景
    frame: false, // 隐藏默认标题栏
    titleBarStyle: 'customButtonsOnHover', // 自定义标题栏样式
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false,
      enableRemoteModule: true,
      backgroundThrottling: false,
      offscreen: false,
      experimentalFeatures: false,
    },
    icon: path.join(__dirname, 'icon.ico'),
    show: false, // 先不显示窗口
  });

  // 加载主页面
  mainWindow.loadFile('index.html');

  // 窗口准备好后显示
  mainWindow.once('ready-to-show', () => {
    mainWindow.show();
    mainWindow.center(); // 再次确保居中
  });

  // 开发环境打开开发者工具
  if (isDev) {
    mainWindow.webContents.openDevTools();
  }

  // 关闭窗口时的处理
  mainWindow.on('closed', () => {
    mainWindow = null;
  });
}

// 应用准备好后创建窗口
app.whenReady().then(async () => {
  // 初始化运行时保护 (已禁用，避免模块缺失错误)
  // try {
  //   protection = new RuntimeProtection();
  //   console.log('运行时保护已启动');
  // } catch (error) {
  //   console.error('运行时保护启动失败:', error);
  // }

  // 设置应用用户数据目录
  const userDataPath = app.getPath('userData');
  console.log(`应用数据目录: ${userDataPath}`);

  // 加载配置
  loadSettings();

  // 创建窗口
  createWindow();
  
  // 在macOS上点击应用图标时重新创建窗口
  app.on('activate', () => {
    if (BrowserWindow.getAllWindows().length === 0) {
      createWindow();
    }
  });
  
  // 添加关于菜单(macOS)
  if (process.platform === 'darwin') {
    app.setAboutPanelOptions({
      applicationName: 'Cursor续杯工具',
      applicationVersion: app.getVersion(),
      version: app.getVersion(),
      copyright: '© 2023 All Rights Reserved'
    });
  }
});

// 所有窗口关闭时退出应用（macOS除外）
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

// 应用退出前清理监控器
app.on('before-quit', () => {
  console.log('🛑 应用即将退出，清理设备ID保护监控...');
  stopDeviceIdProtection();
});

// 定义默认设置
const defaultSettings = {
  autoCheckUpdate: true,
  autoActivateOnStartup: false,
  debugMode: false,
  forceModifyMode: false,
  customCursorPath: '', // 自定义Cursor安装路径
  bypassRegionRestriction: false // 突破地区限制
};

// 当前设置
let currentSettings = { ...defaultSettings };

// 设置文件路径
const settingsPath = path.join(app.getPath('userData'), 'settings.json');

// 加载设置
function loadSettings() {
  try {
    if (fs.existsSync(settingsPath)) {
      const data = fs.readFileSync(settingsPath, 'utf8');
      currentSettings = { ...defaultSettings, ...JSON.parse(data) };
      console.log('加载设置:', currentSettings);
      
      // 应用设置
      applySettings();
    } else {
      // 如果设置文件不存在，创建默认设置
      saveSettings(currentSettings);
    }
  } catch (error) {
    console.error('加载设置失败:', error);
  }
}

// 保存设置
function saveSettings(settings) {
  try {
    fs.writeFileSync(settingsPath, JSON.stringify(settings, null, 2), 'utf8');
    currentSettings = { ...settings };
    console.log('保存设置:', currentSettings);
    
    // 应用设置
    applySettings();
  } catch (error) {
    console.error('保存设置失败:', error);
  }
}

// 应用设置
function applySettings() {
  // 设置环境变量
  if (currentSettings.forceModifyMode) {
    process.env.FORCE_CURSOR_MODIFY = 'true';
    console.log('已启用强制修改模式');
  } else {
    process.env.FORCE_CURSOR_MODIFY = 'false';
  }
  
  // 设置调试模式
  if (currentSettings.debugMode) {
    process.env.DEBUG_MODE = 'true';
    console.log('已启用调试模式');
  } else {
    process.env.DEBUG_MODE = 'false';
  }
}

// IPC 处理程序：获取设置
ipcMain.handle('get-settings', () => {
  return currentSettings;
});

// IPC 处理程序：保存设置
ipcMain.handle('save-settings', (event, settings) => {
  try {
    console.log('收到保存设置请求:', settings);

    // 验证设置对象
    if (!settings || typeof settings !== 'object') {
      throw new Error('无效的设置对象');
    }

    // 确保所有必需的字段都存在
    const validatedSettings = {
      autoCheckUpdate: Boolean(settings.autoCheckUpdate),
      autoActivateOnStartup: Boolean(settings.autoActivateOnStartup),
      debugMode: Boolean(settings.debugMode),
      forceModifyMode: Boolean(settings.forceModifyMode),
      customCursorPath: String(settings.customCursorPath || ''),
      bypassRegionRestriction: Boolean(settings.bypassRegionRestriction)
    };

    saveSettings(validatedSettings);
    console.log('设置保存成功');
    return true;
  } catch (error) {
    console.error('保存设置失败:', error);
    throw error;
  }
});

// IPC 处理程序：检查地区限制突破状态
ipcMain.handle('check-region-restriction-status', async () => {
  try {
    // 构建settings.json路径
    let settingsPath = '';
    if (process.platform === 'win32') {
      settingsPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'settings.json');
    } else if (process.platform === 'darwin') {
      settingsPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'settings.json');
    } else {
      settingsPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'settings.json');
    }

    console.log(`检查 Cursor 设置文件: ${settingsPath}`);

    // 如果文件不存在，返回未启用
    if (!fs.existsSync(settingsPath)) {
      console.log('Cursor 设置文件不存在');
      return { success: true, enabled: false };
    }

    // 读取设置文件
    const settingsContent = fs.readFileSync(settingsPath, 'utf8');
    const currentSettings = JSON.parse(settingsContent);

    // 检查是否包含代理配置的关键字段
    const hasProxy = currentSettings['http.proxy'] === 'socks5://xc999:xc123@154.201.91.204:38999';
    const hasProxySupport = currentSettings['http.proxySupport'] === 'override';
    const hasDisableHttp2 = currentSettings['cursor.general.disableHttp2'] === true;
    const hasSystemCerts = currentSettings['http.systemCertificates'] === false;

    // 只有当所有关键字段都匹配时，才认为是开启状态
    const isEnabled = hasProxy && hasProxySupport && hasDisableHttp2 && hasSystemCerts;

    console.log('地区限制突破状态检查结果:', {
      hasProxy,
      hasProxySupport,
      hasDisableHttp2,
      hasSystemCerts,
      isEnabled
    });

    return {
      success: true,
      enabled: isEnabled,
      currentSettings
    };

  } catch (error) {
    console.error('检查地区限制突破状态失败:', error);
    return { success: false, enabled: false, error: error.message };
  }
});

// IPC 处理程序：切换地区限制突破
ipcMain.handle('toggle-region-restriction', async (event, enabled) => {
  try {
    console.log(`${enabled ? '开启' : '关闭'}地区限制突破...`);

    // 构建settings.json路径
    let settingsPath = '';
    if (process.platform === 'win32') {
      // Windows: %APPDATA%\Cursor\User\settings.json
      settingsPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'settings.json');
    } else if (process.platform === 'darwin') {
      // macOS: ~/Library/Application Support/Cursor/User/settings.json
      settingsPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'settings.json');
    } else {
      // Linux: ~/.config/Cursor/User/settings.json
      settingsPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'settings.json');
    }

    console.log(`Cursor设置文件路径: ${settingsPath}`);

    // 确保目录存在
    const settingsDir = path.dirname(settingsPath);
    if (!fs.existsSync(settingsDir)) {
      fs.mkdirSync(settingsDir, { recursive: true });
      console.log(`已创建设置目录: ${settingsDir}`);
    }

    // 创建备份
    if (fs.existsSync(settingsPath)) {
      const backupPath = `${settingsPath}.bypass_backup`;
      fs.copyFileSync(settingsPath, backupPath);
      console.log(`已创建设置文件备份: ${backupPath}`);
    }

    // 定义开启时的设置
    const enabledSettings = {
      "database-client.autoSync": true,
      "update.enableWindowsBackgroundUpdates": false,
      "update.mode": "none",
      "http.proxyAuthorization": null,
      "json.schemas": [],
      "window.commandCenter": true,
      "http.proxy": "socks5://xc999:xc123@154.201.91.204:38999",
      "http.systemCertificates": false,
      "http.proxySupport": "override",
      "http.experimental.systemCertificatesV2": false,
      "http.experimental.systemCertificates": false,
      "cursor.general.disableHttp2": true
    };

    // 定义关闭时的设置
    const disabledSettings = {
      "database-client.autoSync": true,
      "update.enableWindowsBackgroundUpdates": false,
      "update.mode": "none",
      "http.proxyAuthorization": null,
      "json.schemas": []
    };

    // 根据enabled参数选择设置
    const newSettings = enabled ? enabledSettings : disabledSettings;

    // 写入新设置
    fs.writeFileSync(settingsPath, JSON.stringify(newSettings, null, 4), 'utf8');
    console.log(`已${enabled ? '开启' : '关闭'}地区限制突破`);

    return {
      success: true,
      message: `已${enabled ? '开启' : '关闭'}地区限制突破`,
      settingsPath,
      settings: newSettings
    };

  } catch (error) {
    console.error('切换地区限制突破失败:', error);
    return { success: false, error: error.message };
  }
});

// 从后端API获取当前代理配置
async function fetchCurrentProxyConfig() {
  try {
    const apiUrl = 'http://129.204.108.166:2486/csk/proxy-config/current';
    console.log('📡 从后端获取当前代理配置...');
    
    const response = await axios.get(apiUrl, { 
      timeout: 10000,
      headers: {
        'Content-Type': 'application/json'
      }
    });
    
    if (response.data.success && response.data.data) {
      console.log('✅ 成功获取代理配置');
      console.log('🔗 代理地址:', response.data.data['http.proxy']);
      return response.data.data;
    } else {
      console.warn('⚠️ 获取代理配置失败:', response.data.message);
      return null;
    }
  } catch (error) {
    console.error('❌ 获取代理配置出错:', error.message);
    return null;
  }
}

// 从后端API获取所有启用的代理地址列表
async function fetchAllProxyUrls() {
  try {
    const apiUrl = 'http://129.204.108.166:2486/csk/proxy-config/proxy-urls';
    console.log('📡 从数据库获取所有代理地址列表...');
    
    const response = await axios.get(apiUrl, { 
      timeout: 5000,
      headers: {
        'Content-Type': 'application/json'
      }
    });
    
    if (response.data.success && response.data.data) {
      console.log(`✅ 成功获取 ${response.data.count} 个代理地址`);
      console.log('📋 代理列表:', response.data.data);
      return response.data.data;
    } else {
      console.warn('⚠️ 获取代理列表失败:', response.data.message);
      return null;
    }
  } catch (error) {
    console.error('❌ 获取代理列表出错:', error.message);
    return null;
  }
}

// 获取Cursor settings.json路径
function getCursorSettingsPath() {
  let settingsPath = '';
  if (process.platform === 'win32') {
    settingsPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'settings.json');
  } else if (process.platform === 'darwin') {
    settingsPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'settings.json');
  } else {
    settingsPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'settings.json');
  }
  return settingsPath;
}

// 应用代理配置到Cursor settings.json
async function applyProxyConfigToCursor(proxyConfig) {
  try {
    const settingsPath = getCursorSettingsPath();
    console.log(`📝 应用代理配置到: ${settingsPath}`);

    // 确保目录存在
    const settingsDir = path.dirname(settingsPath);
    if (!fs.existsSync(settingsDir)) {
      fs.mkdirSync(settingsDir, { recursive: true });
      console.log(`已创建设置目录: ${settingsDir}`);
    }

    // 创建备份
    if (fs.existsSync(settingsPath)) {
      const backupPath = `${settingsPath}.auto_switch_backup`;
      fs.copyFileSync(settingsPath, backupPath);
      console.log(`✅ 已创建设置文件备份`);
    }

    // 写入新配置
    fs.writeFileSync(settingsPath, JSON.stringify(proxyConfig, null, 4), 'utf8');
    console.log('✅ 代理配置已成功应用');
    console.log('🔗 当前代理:', proxyConfig['http.proxy']);

    // 发送通知给前端
    if (mainWindow && !mainWindow.isDestroyed()) {
      mainWindow.webContents.send('proxy-config-updated', {
        proxyUrl: proxyConfig['http.proxy'],
        timestamp: new Date().toLocaleString('zh-CN')
      });
    }

    return true;
  } catch (error) {
    console.error('❌ 应用代理配置失败:', error);
    return false;
  }
}

// 执行代理配置切换
async function executeProxySwitch() {
  console.log('🔄 开始执行代理配置自动切换...');
  console.log(`⏰ 时间: ${new Date().toLocaleString('zh-CN')}`);

  try {
    // 从后端获取最新配置
    const proxyConfig = await fetchCurrentProxyConfig();

    if (!proxyConfig) {
      console.warn('⚠️ 无法获取代理配置，使用默认配置');
      // 使用默认配置作为fallback
      const defaultConfig = {
        "database-client.autoSync": true,
        "update.enableWindowsBackgroundUpdates": false,
        "update.mode": "none",
        "http.proxyAuthorization": null,
        "json.schemas": [],
        "window.commandCenter": true,
        "http.proxy": "socks5://xc999:xc123@154.201.91.204:38999",
        "http.systemCertificates": false,
        "http.proxySupport": "override",
        "http.experimental.systemCertificatesV2": false,
        "http.experimental.systemCertificates": false,
        "cursor.general.disableHttp2": true
      };
      await applyProxyConfigToCursor(defaultConfig);
    } else {
      // 应用从数据库获取的配置
      await applyProxyConfigToCursor(proxyConfig);
    }

    console.log('✅ 代理配置切换完成');
    console.log(`⏱️ 下次切换时间: ${new Date(Date.now() + PROXY_SWITCH_INTERVAL).toLocaleString('zh-CN')}`);

  } catch (error) {
    console.error('❌ 代理配置切换失败:', error);
  }
}

// 启动代理自动切换定时器
function startProxyAutoSwitch() {
  if (proxyAutoSwitchTimer) {
    console.log('⚠️ 代理自动切换已在运行中');
    return;
  }

  console.log('🚀 启动代理自动切换功能');
  console.log(`⏰ 切换间隔: ${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60} 小时`);

  isProxyAutoSwitchEnabled = true;

  // 立即执行一次切换
  executeProxySwitch();

  // 设置定时器，每12小时执行一次
  proxyAutoSwitchTimer = setInterval(() => {
    executeProxySwitch();
  }, PROXY_SWITCH_INTERVAL);

  console.log('✅ 代理自动切换定时器已启动');
}

// 恢复代理自动切换定时器（启动时使用，不立即执行切换）
function restoreProxyAutoSwitch() {
  if (proxyAutoSwitchTimer) {
    console.log('⚠️ 代理自动切换已在运行中');
    return;
  }

  console.log('🔄 恢复代理自动切换功能（启动时恢复）');
  console.log(`⏰ 切换间隔: ${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60} 小时`);

  isProxyAutoSwitchEnabled = true;

  // 不立即执行切换，保持当前代理
  // 只设置定时器，12小时后才会执行下一次切换
  proxyAutoSwitchTimer = setInterval(() => {
    executeProxySwitch();
  }, PROXY_SWITCH_INTERVAL);

  console.log('✅ 代理自动切换定时器已恢复（下次切换时间: ' + new Date(Date.now() + PROXY_SWITCH_INTERVAL).toLocaleString('zh-CN') + '）');
}

// 停止代理自动切换定时器并清除代理配置
async function stopProxyAutoSwitch() {
  if (proxyAutoSwitchTimer) {
    clearInterval(proxyAutoSwitchTimer);
    proxyAutoSwitchTimer = null;
    console.log('⏹️ 代理自动切换定时器已停止');
  }
  isProxyAutoSwitchEnabled = false;

  // 清除代理配置，恢复到基础配置
  try {
    const settingsPath = getCursorSettingsPath();
    console.log('🧹 清除代理配置，恢复基础设置...');

    // 基础配置（无代理）
    const basicConfig = {
      "database-client.autoSync": true,
      "update.enableWindowsBackgroundUpdates": false,
      "update.mode": "none",
      "http.proxyAuthorization": null,
      "json.schemas": []
    };

    // 创建备份
    if (fs.existsSync(settingsPath)) {
      const backupPath = `${settingsPath}.disable_backup`;
      fs.copyFileSync(settingsPath, backupPath);
      console.log('✅ 已创建配置备份');
    }

    // 写入基础配置
    fs.writeFileSync(settingsPath, JSON.stringify(basicConfig, null, 4), 'utf8');
    console.log('✅ 已恢复基础配置（已清除代理设置）');

    // 发送通知给前端
    if (mainWindow && !mainWindow.isDestroyed()) {
      mainWindow.webContents.send('proxy-config-cleared');
    }
  } catch (error) {
    console.error('❌ 清除代理配置失败:', error);
  }
}

// IPC: 切换代理自动切换功能
ipcMain.handle('toggle-proxy-auto-switch', async (event, enabled) => {
  try {
    console.log(`${enabled ? '开启' : '关闭'}代理自动切换功能`);

    if (enabled) {
      startProxyAutoSwitch();
      return {
        success: true,
        message: '代理自动切换已开启',
        interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`
      };
    } else {
      await stopProxyAutoSwitch();
      return {
        success: true,
        message: '代理自动切换已关闭，已清除代理配置'
      };
    }
  } catch (error) {
    console.error('切换代理自动切换功能失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// IPC: 获取代理自动切换状态
ipcMain.handle('get-proxy-auto-switch-status', async () => {
  try {
    console.log('🔍 检查代理自动切换状态...');
    
    // 1. 读取 settings.json 中的当前代理
    const settingsPath = getCursorSettingsPath();
    let currentProxy = null;
    let hasProxyConfig = false;
    
    if (fs.existsSync(settingsPath)) {
      try {
        const fileContent = fs.readFileSync(settingsPath, 'utf8');
        const settings = JSON.parse(fileContent);
        currentProxy = settings['http.proxy'];
        
        // 检查是否有完整的代理配置
        hasProxyConfig = currentProxy && 
                         currentProxy.length > 0 &&
                         settings['http.proxySupport'] === 'override' &&
                         settings['cursor.general.disableHttp2'] === true;
        
        if (currentProxy) {
          console.log(`📍 settings.json中的代理: ${currentProxy}`);
        } else {
          console.log('📍 settings.json中没有代理配置');
        }
      } catch (error) {
        console.error('❌ 读取settings.json失败:', error.message);
      }
    } else {
      console.log('📍 settings.json文件不存在');
    }
    
    // 2. 如果没有代理配置，直接返回关闭状态
    if (!hasProxyConfig) {
      console.log('⚪ 没有完整的代理配置，返回关闭状态');
      return {
        success: true,
        enabled: false,
        interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
        message: '未配置代理'
      };
    }
    
    // 3. 从数据库获取所有启用的代理地址列表
    const proxyUrls = await fetchAllProxyUrls();
    
    if (proxyUrls && proxyUrls.length > 0) {
      // 4. 比对当前代理是否在数据库列表中
      const isInDatabase = proxyUrls.includes(currentProxy);
      
      if (isInDatabase) {
        console.log('✅ 当前代理在数据库列表中');
        console.log('🟢 状态: 开启');
        
        // 如果定时器还没启动，自动恢复（不立即切换）
        if (!isProxyAutoSwitchEnabled && !proxyAutoSwitchTimer) {
          console.log('🔄 自动恢复代理自动切换功能...');
          restoreProxyAutoSwitch();
        }
        
        return {
          success: true,
          enabled: true,  // 代理在数据库中，返回开启状态
          interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
          nextSwitchTime: proxyAutoSwitchTimer ? new Date(Date.now() + PROXY_SWITCH_INTERVAL).toLocaleString('zh-CN') : null,
          currentProxy: currentProxy,
          inDatabase: true
        };
      } else {
        console.log('❌ 当前代理不在数据库列表中');
        console.log(`当前代理: ${currentProxy}`);
        console.log('可能原因：');
        console.log('  1. 代理配置被手动修改');
        console.log('  2. 数据库中已禁用此代理');
        console.log('  3. 使用了非数据库管理的代理');
        console.log('⚪ 状态: 关闭');
        
        // 代理不在数据库中，返回关闭状态
        return {
          success: true,
          enabled: false,
          interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
          message: '当前代理不在数据库中',
          currentProxy: currentProxy,
          inDatabase: false
        };
      }
    } else {
      // 5. 无法获取数据库列表，使用fallback逻辑
      console.warn('⚠️ 无法从数据库获取代理列表，使用fallback逻辑');
      
      // Fallback: 如果有完整的代理配置，认为是开启的（但未验证数据库）
      if (hasProxyConfig) {
        console.log('⚠️ 有代理配置但未验证数据库，返回开启状态');
        
        // 如果定时器还没启动，自动恢复（fallback模式，不立即切换）
        if (!isProxyAutoSwitchEnabled && !proxyAutoSwitchTimer) {
          console.log('🔄 Fallback模式：自动恢复代理自动切换功能...');
          restoreProxyAutoSwitch();
        }
        
        return {
          success: true,
          enabled: true,  // 有完整配置，返回开启状态
          interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
          nextSwitchTime: proxyAutoSwitchTimer ? new Date(Date.now() + PROXY_SWITCH_INTERVAL).toLocaleString('zh-CN') : null,
          currentProxy: currentProxy,
          fallback: true,
          message: '代理已开启（未验证数据库）'
        };
      } else {
        return {
          success: true,
          enabled: false,
          interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
          message: '未配置代理'
        };
      }
    }
  } catch (error) {
    console.error('❌ 获取代理状态失败:', error);
    return {
      success: true,
      enabled: false,
      interval: `${PROXY_SWITCH_INTERVAL / 1000 / 60 / 60}小时`,
      message: '检查失败，默认关闭'
    };
  }
});

// IPC: 手动触发代理配置切换
ipcMain.handle('manual-proxy-switch', async () => {
  try {
    console.log('🔄 手动触发代理配置切换...');
    await executeProxySwitch();
    return {
      success: true,
      message: '代理配置已更新'
    };
  } catch (error) {
    console.error('手动切换代理失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// IPC: 卡密验证失败时清除代理配置
ipcMain.handle('clear-proxy-on-card-invalid', async () => {
  try {
    console.log('⚠️ 卡密验证失败，清除代理配置...');
    
    const settingsPath = getCursorSettingsPath();
    
    // 基础配置（无代理）
    const basicConfig = {
      "database-client.autoSync": true,
      "update.enableWindowsBackgroundUpdates": false,
      "update.mode": "none",
      "http.proxyAuthorization": null,
      "json.schemas": []
    };

    // 确保目录存在
    const settingsDir = path.dirname(settingsPath);
    if (!fs.existsSync(settingsDir)) {
      fs.mkdirSync(settingsDir, { recursive: true });
    }

    // 创建备份
    if (fs.existsSync(settingsPath)) {
      const backupPath = `${settingsPath}.card_invalid_backup`;
      fs.copyFileSync(settingsPath, backupPath);
      console.log('✅ 已创建配置备份');
    }

    // 写入基础配置
    fs.writeFileSync(settingsPath, JSON.stringify(basicConfig, null, 4), 'utf8');
    console.log('✅ 已清除代理配置，恢复基础设置');

    // 停止定时器
    if (proxyAutoSwitchTimer) {
      clearInterval(proxyAutoSwitchTimer);
      proxyAutoSwitchTimer = null;
      isProxyAutoSwitchEnabled = false;
      console.log('⏹️ 已停止代理自动切换定时器');
    }

    // 发送通知给前端
    if (mainWindow && !mainWindow.isDestroyed()) {
      mainWindow.webContents.send('proxy-config-cleared');
    }

    return {
      success: true,
      message: '代理配置已清除'
    };
  } catch (error) {
    console.error('❌ 清除代理配置失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// IPC 处理程序：禁用Cursor自动更新
ipcMain.handle('disable-cursor-auto-update', async () => {
  try {
    console.log('正在禁用Cursor自动更新...');

    // 构建settings.json路径
    let settingsPath = '';
    if (process.platform === 'win32') {
      // Windows: %APPDATA%\Cursor\User\settings.json
      settingsPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'settings.json');
    } else if (process.platform === 'darwin') {
      // macOS: ~/Library/Application Support/Cursor/User/settings.json
      settingsPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'settings.json');
    } else {
      // Linux: ~/.config/Cursor/User/settings.json
      settingsPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'settings.json');
    }

    console.log(`Cursor设置文件路径: ${settingsPath}`);

    // 确保目录存在
    const settingsDir = path.dirname(settingsPath);
    if (!fs.existsSync(settingsDir)) {
      fs.mkdirSync(settingsDir, { recursive: true });
      console.log(`已创建设置目录: ${settingsDir}`);
    }

    // 读取现有设置
    let currentSettings = {};
    if (fs.existsSync(settingsPath)) {
      try {
        const settingsContent = fs.readFileSync(settingsPath, 'utf8');
        currentSettings = JSON.parse(settingsContent);
        console.log('已读取现有设置');
      } catch (parseError) {
        console.warn('解析现有设置失败，将创建新设置:', parseError.message);
        currentSettings = {};
      }
    }

    // 添加禁用自动更新的设置
    currentSettings['update.enableWindowsBackgroundUpdates'] = false;
    currentSettings['update.mode'] = 'none';

    // 创建备份
    if (fs.existsSync(settingsPath)) {
      const backupPath = `${settingsPath}.bak`;
      fs.copyFileSync(settingsPath, backupPath);
      console.log(`已创建设置文件备份: ${backupPath}`);
    }

    // 写入新设置
    fs.writeFileSync(settingsPath, JSON.stringify(currentSettings, null, 2), 'utf8');
    console.log('已成功禁用Cursor自动更新');

    return {
      success: true,
      message: '已成功禁用Cursor自动更新',
      settingsPath,
      settings: currentSettings
    };

  } catch (error) {
    console.error('禁用Cursor自动更新失败:', error);
    return { success: false, error: error.message };
  }
});

// 获取机器ID
ipcMain.handle('get-machine-id', async () => {
  try {
    let machineId = '';

    if (process.platform === 'win32') {
      // Windows: 使用WMIC获取UUID
      machineId = await new Promise((resolve, reject) => {
        exec('wmic csproduct get uuid', (error, stdout) => {
          if (error) {
            reject(error);
            return;
          }
          const uuid = stdout.split('\n')[1].trim();
          resolve(uuid);
        });
      });
    } else if (process.platform === 'darwin') {
      // macOS: 使用ioreg获取IOPlatformUUID
      machineId = await new Promise((resolve, reject) => {
        exec('ioreg -rd1 -c IOPlatformExpertDevice | grep -i "IOPlatformUUID" | awk \'{print $3}\' | sed -e s/\\\"//g', (error, stdout) => {
          if (error) {
            reject(error);
            return;
          }
          resolve(stdout.trim());
        });
      });
    } else {
      // Linux: 使用/etc/machine-id或/var/lib/dbus/machine-id
      const machineIdPath = fs.existsSync('/etc/machine-id') ? '/etc/machine-id' : '/var/lib/dbus/machine-id';
      machineId = fs.readFileSync(machineIdPath, 'utf8').trim();
    }

    return machineId;
  } catch (error) {
    console.error('获取机器ID失败:', error);
    // 如果获取失败，使用网络接口MAC地址作为备选
    const networkInterfaces = os.networkInterfaces();
    for (const name of Object.keys(networkInterfaces)) {
      for (const net of networkInterfaces[name]) {
        // 跳过内部IP和非物理接口
        if (!net.internal && net.mac && net.mac !== '00:00:00:00:00:00') {
          return net.mac;
        }
      }
    }
    return 'unknown-device-id';
  }
});

// 获取Cursor用户数据路径
ipcMain.handle('get-user-data-path', async () => {
  try {
    let userDataPath = '';

    if (process.platform === 'win32') {
      // Windows: %APPDATA%\Cursor
      userDataPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor');
    } else if (process.platform === 'darwin') {
      // macOS: ~/Library/Application Support/Cursor
      userDataPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor');
    } else {
      // Linux: ~/.config/Cursor
      userDataPath = path.join(os.homedir(), '.config', 'Cursor');
    }

    console.log(`Cursor用户数据路径: ${userDataPath}`);
    return userDataPath;
  } catch (error) {
    console.error('获取用户数据路径失败:', error);
    throw error;
  }
});

// 打开文件对话框
ipcMain.handle('open-file-dialog', async () => {
  const result = await dialog.showOpenDialog(mainWindow, {
    properties: ['openFile'],
    filters: [
      { name: 'Cursor配置文件', extensions: ['json'] }
    ]
  });

  if (!result.canceled && result.filePaths.length > 0) {
    return result.filePaths[0];
  }
  return null;
});

// 打开文件夹对话框
ipcMain.handle('open-folder-dialog', async () => {
  const result = await dialog.showOpenDialog(mainWindow, {
    properties: ['openDirectory'],
    title: '选择Cursor安装目录'
  });

  if (!result.canceled && result.filePaths.length > 0) {
    return result.filePaths[0];
  }
  return null;
});

// 获取最新公告
ipcMain.handle('get-latest-notice', async () => {
  try {
    console.log('正在获取最新公告...');

    // 使用新的 Render 后端 API（与 Mac 版本保持一致）
    const config = require('./config');
    const apiUrl = `${config.API_BASE_URL}/api/v1/announcement`;

    // 首先尝试从真实API获取数据
    try {
      console.log('尝试连接后端API:', apiUrl);
      const result = await makeHttpRequest(apiUrl);
      if (result && result.success && result.data) {
        console.log('从API获取公告成功:', result.data);
        return result;
      } else if (result) {
        console.log('API返回数据格式异常:', result);
      }
    } catch (apiError) {
      console.log('API请求失败，使用模拟数据:', apiError.message);
    }

    // 如果API不可用，返回模拟数据
    console.log('返回模拟的公告数据');

    // 模拟最新公告数据
    return {
      success: true,
      data: {
        id: 1,
        title: '最新公告',
        content: '基于Python项目逻辑重构，使用数据库操作方式实现账号切换，更加稳定可靠。如果遇到问题请联系开发者。',
        time: new Date().toISOString().replace('T', ' ').substring(0, 16),
        user: '系统管理员'
      }
    };


  } catch (error) {
    console.error('获取公告失败:', error);

    // 返回默认公告
    return {
      success: false,
      error: error.message,
      data: {
        id: 0,
        title: '系统公告',
        content: '基于Python项目逻辑重构，使用数据库操作方式实现账号切换，更加稳定可靠。如果遇到问题请联系开发者。',
        time: '2024-07-06 17:10',
        user: '系统'
      }
    };
  }
});

// 读取文件内容
ipcMain.handle('read-file', async (event, filePath) => {
  try {
    const content = fs.readFileSync(filePath, 'utf8');
    return content;
  } catch (error) {
    console.error('读取文件失败:', error);
    return null;
  }
});

// 写入文件
ipcMain.handle('write-file', async (event, filePath, content) => {
  try {
    fs.writeFileSync(filePath, content);
    console.log(`成功写入文件: ${filePath}`);
    return true;
  } catch (error) {
    console.error(`写入文件失败 ${filePath}:`, error);
    return false;
  }
});

// 获取Cursor路径的内部函数
async function getCursorPaths() {
  try {
    let basePath, packagePath, mainPath, workbenchPath;

    // 首先检查是否有自定义路径
    if (currentSettings.customCursorPath && currentSettings.customCursorPath.trim()) {
      // 使用自定义路径
      const customPath = currentSettings.customCursorPath.trim();
      console.log(`使用自定义Cursor路径: ${customPath}`);

      // 检查自定义路径是否是Cursor安装目录
      if (fs.existsSync(path.join(customPath, 'Cursor.exe')) ||
          fs.existsSync(path.join(customPath, 'Cursor.app')) ||
          fs.existsSync(path.join(customPath, 'cursor'))) {
        // 这是Cursor的安装根目录，需要找到resources/app
        if (process.platform === 'win32') {
          basePath = path.join(customPath, 'resources', 'app');
        } else if (process.platform === 'darwin') {
          basePath = path.join(customPath, 'Contents', 'Resources', 'app');
        } else {
          basePath = path.join(customPath, 'resources', 'app');
        }
      } else if (fs.existsSync(path.join(customPath, 'package.json'))) {
        // 这可能直接是resources/app目录
        basePath = customPath;
      } else {
        // 尝试在自定义路径下查找resources/app
        basePath = path.join(customPath, 'resources', 'app');
      }
    } else {
      // 使用默认路径
      if (process.platform === 'win32') {
        // Windows
        basePath = path.join(os.homedir(), 'AppData', 'Local', 'Programs', 'Cursor', 'resources', 'app');
      } else if (process.platform === 'darwin') {
        // macOS
        basePath = '/Applications/Cursor.app/Contents/Resources/app';
      } else {
        // Linux
        const possiblePaths = [
          '/opt/Cursor/resources/app',
          '/usr/share/cursor/resources/app'
        ];
        for (const p of possiblePaths) {
          if (fs.existsSync(p)) {
            basePath = p;
            break;
          }
        }
      }
    }

    if (!basePath || !fs.existsSync(basePath)) {
      return { error: 'Cursor安装路径不存在' };
    }

    packagePath = path.join(basePath, 'package.json');
    mainPath = path.join(basePath, 'out', 'main.js');
    workbenchPath = path.join(basePath, 'out', 'vs', 'workbench', 'workbench.desktop.main.js');

    if (!fs.existsSync(packagePath) || !fs.existsSync(mainPath)) {
      return { error: 'Cursor核心文件不存在' };
    }

    // 读取版本信息
    let version = '';
    try {
      const packageContent = fs.readFileSync(packagePath, 'utf8');
      version = JSON.parse(packageContent).version;
    } catch (err) {
      console.error('读取版本信息失败:', err);
    }

    return {
      basePath,
      packagePath,
      mainPath,
      workbenchPath,
      version,
      error: null
    };
  } catch (error) {
    console.error('获取Cursor路径失败:', error);
    return { error: error.message };
  }
}

// 获取Cursor相关路径
ipcMain.handle('get-cursor-paths', async () => {
  return await getCursorPaths();
});

// 验证Cursor文件完整性的辅助函数
function validateCursorFile(filePath, content) {
  try {
    // 基本的语法检查
    const openBraces = (content.match(/\{/g) || []).length;
    const closeBraces = (content.match(/\}/g) || []).length;
    const openParens = (content.match(/\(/g) || []).length;
    const closeParens = (content.match(/\)/g) || []).length;

    if (openBraces !== closeBraces) {
      return { valid: false, error: `大括号不匹配: ${openBraces} vs ${closeBraces}` };
    }

    if (openParens !== closeParens) {
      return { valid: false, error: `小括号不匹配: ${openParens} vs ${closeParens}` };
    }

    // 检查是否包含明显的语法错误
    const syntaxErrors = [
      /\!\s*\!/g,  // 双感叹号
      /\?\s*\?\s*\?/g,  // 三个问号
      /return\s*\?\?/g,  // return后直接跟??
      /\{\s*\?\?/g,  // 大括号后直接跟??
    ];

    for (const errorPattern of syntaxErrors) {
      if (errorPattern.test(content)) {
        return { valid: false, error: `检测到语法错误模式: ${errorPattern.source}` };
      }
    }

    return { valid: true };
  } catch (error) {
    return { valid: false, error: `验证过程出错: ${error.message}` };
  }
}

// 基于Python项目逻辑的简化版本 - 修改Cursor的main.js文件
ipcMain.handle('modify-cursor-main-js', async (event, mainPath) => {
  try {
    if (!fs.existsSync(mainPath)) {
      return { success: false, error: '文件不存在' };
    }

    // 创建备份
    const backupPath = `${mainPath}.bak`;
    if (!fs.existsSync(backupPath)) {
      fs.copyFileSync(mainPath, backupPath);
      console.log(`已创建备份: ${backupPath}`);
    }

    // 读取文件内容
    const content = fs.readFileSync(mainPath, 'utf8');

    // 添加调试日志
    console.log('开始修改main.js文件...');
    console.log('文件大小:', content.length, '字节');

    // 检查是否启用强制模式
    const forceMode = process.env.FORCE_CURSOR_MODIFY === 'true';
    console.log('强制修改模式:', forceMode ? '已启用' : '未启用');

    let newContent = content;
    let modified = false;

    // 使用Python项目中的简化正则表达式模式
    const pythonStylePatterns = [
      {
        // 基于Python项目的模式1: async getMachineId(){return [^??]+??([^}]+)}
        regex: /async getMachineId\(\)\{return [^??]+\?\?([^}]+)\}/g,
        replacement: 'async getMachineId(){return $1}'
      },
      {
        // 基于Python项目的模式2: async getMacMachineId(){return [^??]+??([^}]+)}
        regex: /async getMacMachineId\(\)\{return [^??]+\?\?([^}]+)\}/g,
        replacement: 'async getMacMachineId(){return $1}'
      }
    ];

    // 检查文件内容是否包含getMachineId函数
    if (content.includes('getMachineId')) {
      console.log('文件包含getMachineId关键词');

      // 提取并记录getMachineId函数上下文
      const contextRegex = /[\s\S]{0,100}getMachineId[\s\S]{0,100}/g;
      const contexts = content.match(contextRegex);
      if (contexts && contexts.length > 0) {
        console.log('找到的getMachineId上下文:');
        contexts.forEach((ctx, i) => {
          console.log(`上下文 ${i + 1}:`, ctx.replace(/\n/g, '\\n'));
        });
      }
    } else {
      console.log('文件不包含getMachineId关键词');
    }

    // 应用Python项目风格的简单替换
    for (const pattern of pythonStylePatterns) {
      if (pattern.regex.test(newContent)) {
        console.log('匹配到Python风格模式:', pattern.regex.source);
        const beforeReplace = newContent;
        newContent = newContent.replace(pattern.regex, pattern.replacement);

        // 验证替换是否成功
        if (newContent !== beforeReplace) {
          console.log('Python风格模式替换成功');
          modified = true;
          break; // 只应用第一个匹配的模式
        }
      }
    }

    // 如果Python风格模式失败，尝试更宽松的模式（但仍然安全）
    if (!modified) {
      console.log('尝试宽松模式...');

      // 宽松模式：直接查找并替换??操作符
      const loosePattern = /(getMachineId[^{]*\{[^}]*?return\s+[^;]*?)(\?\?)([^;}]+)/g;

      if (loosePattern.test(newContent)) {
        const beforeReplace = newContent;
        newContent = newContent.replace(loosePattern, (match, prefix, nullOp, afterNull) => {
          console.log('匹配到宽松模式:', match.substring(0, 50) + '...');

          // 验证afterNull部分是否安全
          const cleanAfterNull = afterNull.trim();
          if (cleanAfterNull && !cleanAfterNull.includes('{') && !cleanAfterNull.includes('}')) {
            return `${prefix}${cleanAfterNull}`;
          } else {
            console.log('跳过不安全的替换');
            return match; // 保持原样
          }
        });

        if (newContent !== beforeReplace) {
          console.log('宽松模式替换成功');
          modified = true;
        }
      }
    }

    // 最后的安全检查：验证修改后的代码不包含明显的语法错误
    if (modified) {
      console.log('执行语法安全检查...');

      const validation = validateCursorFile(mainPath, newContent);
      if (!validation.valid) {
        console.error(`语法验证失败: ${validation.error}`);
        console.log('恢复原始内容...');
        newContent = content; // 恢复原始内容
        modified = false;
      } else {
        console.log('语法安全检查通过');
      }
    }

    // 如果所有尝试都失败，提供强制成功选项（但要更加谨慎）
    if (!modified && process.env.FORCE_CURSOR_MODIFY === 'true') {
      console.log('强制修改模式开启，尝试最小化修改...');

      // 在强制模式下，只进行最基本的替换，避免破坏语法
      const forcePattern = /(\?\?\s*[^;}]+)/g;
      const matches = newContent.match(forcePattern);

      if (matches && matches.length > 0) {
        console.log(`强制模式：找到${matches.length}个??操作符`);

        // 只替换明确包含getMachineId上下文的??操作符
        const contextPattern = /(getMachineId[^{]*\{[^}]*?)(\?\?\s*)([^;}]+)/g;
        const beforeForce = newContent;
        newContent = newContent.replace(contextPattern, (match, prefix, nullOp, fallback) => {
          console.log('强制模式替换:', match.substring(0, 30) + '...');
          return `${prefix}${fallback.trim()}`;
        });

        if (newContent !== beforeForce) {
          modified = true;
          console.log('强制模式替换成功');
        }
      }

      // 如果强制模式仍然失败，则不修改文件
      if (!modified) {
        console.log('强制模式也无法安全修改文件');
        return { success: false, error: '无法安全修改文件，建议手动修改或使用备份恢复' };
      }
    }

    if (!modified) {
      console.log('所有模式匹配失败');
      return { success: false, error: '未找到匹配的函数模式，建议启用强制修改模式或手动修改' };
    }

    // 最终写入前的安全检查
    try {
      // 尝试解析修改后的内容（简单的语法检查）
      const testContent = newContent.substring(0, Math.min(1000, newContent.length));
      if (testContent.includes('??')) {
        console.warn('警告：修改后的内容仍包含??操作符，可能修改不完整');
      }

      // 写入修改后的内容
      fs.writeFileSync(mainPath, newContent, 'utf8');
      console.log('文件修改完成');

      return { success: true, message: '文件修改成功' };
    } catch (writeError) {
      console.error('写入文件失败:', writeError);
      return { success: false, error: `写入文件失败: ${writeError.message}` };
    }
  } catch (error) {
    console.error('修改main.js文件失败:', error);
    return { success: false, error: error.message };
  }
});

// 分析Cursor文件内容
ipcMain.handle('analyze-cursor-file', async (event, filePath) => {
  try {
    if (!fs.existsSync(filePath)) {
      return { success: false, error: '文件不存在' };
    }

    const content = fs.readFileSync(filePath, 'utf8');
    const analysis = {
      fileSize: content.length,
      containsGetMachineId: content.includes('getMachineId'),
      containsNullishCoalescing: content.includes('??'),
      patterns: []
    };

    // 查找所有可能的getMachineId函数模式
    const searchPatterns = [
      /getMachineId[^{]*\{[^}]*\?\?[^}]*\}/g,
      /async\s+getMachineId[^{]*\{[^}]*\?\?[^}]*\}/g,
      /getMachineId\s*=\s*async[^{]*\{[^}]*\?\?[^}]*\}/g,
      /getMachineId\s*:\s*async[^{]*\{[^}]*\?\?[^}]*\}/g
    ];

    for (let i = 0; i < searchPatterns.length; i++) {
      const matches = content.match(searchPatterns[i]);
      if (matches) {
        analysis.patterns.push({
          patternIndex: i,
          matches: matches.map(match => ({
            content: match.substring(0, 100) + (match.length > 100 ? '...' : ''),
            length: match.length
          }))
        });
      }
    }

    // 查找所有包含??的行
    const lines = content.split('\n');
    const nullishLines = [];
    for (let i = 0; i < lines.length; i++) {
      if (lines[i].includes('??')) {
        nullishLines.push({
          lineNumber: i + 1,
          content: lines[i].trim().substring(0, 100)
        });
      }
    }
    analysis.nullishLines = nullishLines.slice(0, 10); // 只返回前10行

    return { success: true, analysis };
  } catch (error) {
    console.error('分析文件失败:', error);
    return { success: false, error: error.message };
  }
});

// 恢复Cursor文件备份
ipcMain.handle('restore-cursor-backup', async (event, filePath) => {
  try {
    const backupPath = `${filePath}.bak`;

    if (!fs.existsSync(backupPath)) {
      return { success: false, error: '备份文件不存在' };
    }

    if (!fs.existsSync(filePath)) {
      return { success: false, error: '原文件不存在' };
    }

    // 恢复备份
    fs.copyFileSync(backupPath, filePath);
    console.log(`已恢复备份: ${backupPath} -> ${filePath}`);

    return { success: true, message: '备份恢复成功' };
  } catch (error) {
    console.error('恢复备份失败:', error);
    return { success: false, error: error.message };
  }
});

// 修改workbench.desktop.main.js文件
ipcMain.handle('modify-cursor-workbench', async (event, workbenchPath, isValid, days) => {
  try {
    if (!fs.existsSync(workbenchPath)) {
      return { success: false, error: '文件不存在' };
    }
    
    // 创建备份
    const backupPath = `${workbenchPath}.bak`;
    if (!fs.existsSync(backupPath)) {
      fs.copyFileSync(workbenchPath, backupPath);
      console.log(`已创建备份: ${backupPath}`);
    }
    
    // 读取文件内容
    const content = fs.readFileSync(workbenchPath, 'utf8');
    
    // 根据卡密状态构建替换模式
    const patterns = [];
    
    if (isValid) {
      // 启用专业版特权
      patterns.push({
        regex: /(isPro:function\(\)\{return )(.*?)(\})/g,
        replacement: '$1true$3'
      });
      
      // 设置使用天数
      patterns.push({
        regex: /(getCursorTeamInfo:function\(\)\{return\{)([^}]*?)(\}\})/g,
        replacement: `$1usageDays:${days}$3`
      });
    } else {
      // 禁用专业版特权
      patterns.push({
        regex: /(isPro:function\(\)\{return )(.*?)(\})/g,
        replacement: '$1false$3'
      });
    }
    
    let newContent = content;
    let modified = false;
    
    for (const pattern of patterns) {
      if (pattern.regex.test(newContent)) {
        newContent = newContent.replace(pattern.regex, pattern.replacement);
        modified = true;
      }
    }
    
    // 如果所有尝试都失败，提供强制成功选项
    if (!modified && process.env.FORCE_CURSOR_MODIFY === 'true') {
      console.log('强制修改模式开启，跳过workbench.desktop.main.js模式匹配检查');
      modified = true;
      
      // 在强制模式下，尝试直接注入代码
      if (isValid) {
        // 尝试找到函数并替换
        const proFunctionRegex = /isPro\s*:\s*function\s*\(\s*\)\s*\{[^}]*\}/g;
        if (proFunctionRegex.test(newContent)) {
          newContent = newContent.replace(proFunctionRegex, 'isPro:function(){return true}');
          console.log('强制修改isPro函数成功');
        } else {
          // 尝试在文件末尾注入代码
          newContent = newContent + '\n// 强制启用专业版\nwindow.isPro = function() { return true; };\n';
          console.log('注入isPro函数成功');
        }
      }
    }
    
    if (!modified) {
      return { success: false, error: '未找到匹配的函数模式' };
    }
    
    // 写入修改后的内容
    fs.writeFileSync(workbenchPath, newContent, 'utf8');
    
    return { success: true };
  } catch (error) {
    console.error('修改workbench文件失败:', error);
    return { success: false, error: error.message };
  }
});

// 生成新的ID
function generateNewIds() {
  const machineId = uuidv4();
  const anonymousId = uuidv4();
  const machineIdHash = crypto.createHash('md5').update(uuidv4()).digest('hex');
  
  return {
    machineId,
    anonymousId,
    machineIdHash
  };
}

// 更新SQLite数据库中的ID
ipcMain.handle('update-cursor-sqlite-db', async (event, dbPath) => {
  try {
    if (!fs.existsSync(dbPath)) {
      return { success: true, message: '数据库文件不存在，跳过更新' };
    }

    // 使用新的数据库封装
    const db = new CursorDatabase(dbPath);

    // 连接数据库
    const connectResult = db.connect();
    if (!connectResult.success) {
      return connectResult;
    }

    // 如果是模拟模式
    if (connectResult.simulation) {
      return {
        success: true,
        message: '数据库模拟更新成功',
        updatedKeys: ['machineId', 'anonymousId', 'machineIdHash'],
        newIds: generateNewIds()
      };
    }

    // 生成新的ID
    const newIds = generateNewIds();

    // 创建备份
    const backupResult = db.createBackup();
    if (!backupResult.success && !backupResult.simulation) {
      console.warn('创建备份失败，但继续执行:', backupResult.error);
    }

    // 检查表是否存在
    const tableCheck = db.checkTable('ItemTable');
    if (!tableCheck.success) {
      db.close();
      return tableCheck;
    }

    if (!tableCheck.exists && !tableCheck.simulation) {
      db.close();
      return { success: true, message: 'ItemTable表不存在，跳过更新' };
    }

    // 准备更新数据
    const updates = Object.entries(newIds);

    // 批量更新
    const updateResult = db.batchUpdate(updates);

    // 关闭数据库连接
    db.close();

    if (updateResult.success) {
      return {
        success: true,
        message: '数据库更新成功',
        updatedKeys: updateResult.updatedKeys || Object.keys(newIds),
        newIds: newIds
      };
    } else {
      return {
        success: false,
        error: `数据库更新失败: ${updateResult.error}`
      };
    }

  } catch (error) {
    console.error('更新SQLite数据库失败:', error);
    return { success: false, error: error.message };
  }
});

// Mac专用的重置Cursor设备ID处理器
ipcMain.handle('reset-cursor-device-id-mac', async () => {
  try {
    console.log('=== Mac专用设备ID重置开始 ===');
    
    if (process.platform !== 'darwin') {
      return { 
        success: false, 
        error: '此功能仅适用于Mac系统' 
      };
    }

    // 获取Cursor用户数据路径
    const appSupport = path.join(os.homedir(), 'Library', 'Application Support');
    const cursorDir = path.join(appSupport, 'Cursor');
    
    console.log(`Mac Cursor目录: ${cursorDir}`);
    
    if (!fs.existsSync(cursorDir)) {
      return { 
        success: false, 
        error: 'Cursor目录不存在，请确保Cursor已安装' 
      };
    }

    // 执行Mac专用的storage.json重置
    const macResetResult = await resetMacStorageJsonDeviceIds(cursorDir);
    
    // 同时执行传统的机器ID重置作为额外保障
    let traditionalResetResult = { success: true, files: [] };
    try {
      const machineIdFiles = [
        path.join(cursorDir, 'machineid.json'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'anonymousid')
      ];

      const { v4: uuidv4 } = require('uuid');
      const deviceId = uuidv4();
      const anonymousId = uuidv4();

      for (const filePath of machineIdFiles) {
        try {
          const dir = path.dirname(filePath);
          if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
          }

          const fileName = path.basename(filePath);
          const fileExtension = path.extname(filePath);

          if (fileExtension === '.json') {
            const data = { machineId: deviceId };
            fs.writeFileSync(filePath, JSON.stringify(data, null, 2));
            traditionalResetResult.files.push(filePath);
          } else {
            const idToWrite = fileName === 'anonymousid' ? anonymousId : deviceId;
            fs.writeFileSync(filePath, idToWrite);
            traditionalResetResult.files.push(filePath);
          }
        } catch (fileError) {
          console.warn(`处理文件失败: ${filePath}, ${fileError.message}`);
        }
      }
      
      console.log(`传统方式处理了 ${traditionalResetResult.files.length} 个文件`);
    } catch (traditionalError) {
      console.warn('传统重置方式失败:', traditionalError.message);
      traditionalResetResult.success = false;
      traditionalResetResult.error = traditionalError.message;
    }

    // 总结结果
    const overallSuccess = macResetResult.success || traditionalResetResult.success;
    
    return {
      success: overallSuccess,
      message: overallSuccess ? 'Mac设备ID重置成功' : '设备ID重置失败',
      details: {
        storageJsonReset: macResetResult,
        traditionalReset: traditionalResetResult,
        timestamp: new Date().toISOString()
      }
    };

  } catch (error) {
    console.error('Mac设备ID重置失败:', error);
    return { 
      success: false, 
      error: error.message 
    };
  }
});

// 重置机器ID文件
ipcMain.handle('reset-cursor-machine-id', async () => {
  try {
    let idFiles = [];
    let dbPath = '';
    
    // 根据操作系统设置路径
    if (process.platform === 'win32') {
      // Windows
      const appData = process.env.APPDATA;
      dbPath = path.join(appData, 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      idFiles = [
        path.join(appData, 'Cursor', 'machineid.json'),
        path.join(appData, 'Cursor', 'machineid'),
        path.join(appData, 'Cursor', 'User', 'globalStorage', 'machine-id'),
        path.join(appData, 'Cursor', 'User', 'globalStorage', 'anonymousid')
      ];
    } else if (process.platform === 'darwin') {
      // macOS
      const appSupport = path.join(os.homedir(), 'Library', 'Application Support');
      dbPath = path.join(appSupport, 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      idFiles = [
        path.join(appSupport, 'Cursor', 'machineid.json'),
        path.join(appSupport, 'Cursor', 'machineid'),
        path.join(appSupport, 'Cursor', 'User', 'globalStorage', 'machine-id'),
        path.join(appSupport, 'Cursor', 'User', 'globalStorage', 'anonymousid'),
        // macOS 特定的额外清理
        path.join(appSupport, 'Cursor', 'SharedStorage'),
        path.join(appSupport, 'Cursor', 'Trust Tokens'),
        path.join(appSupport, 'Cursor', 'Trust Tokens-journal')
      ];
    } else {
      // Linux
      const configDir = path.join(os.homedir(), '.config');
      dbPath = path.join(configDir, 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      idFiles = [
        path.join(configDir, 'Cursor', 'machineid.json'),
        path.join(configDir, 'Cursor', 'machineid'),
        path.join(configDir, 'Cursor', 'User', 'globalStorage', 'machine-id'),
        path.join(configDir, 'Cursor', 'User', 'globalStorage', 'anonymousid')
      ];
    }
    
    // 删除ID文件
    const deletedFiles = [];
    for (const file of idFiles) {
      try {
        if (fs.existsSync(file)) {
          const stat = fs.statSync(file);
          if (stat.isFile()) {
            fs.unlinkSync(file);
            deletedFiles.push(file);
            console.log(`已删除文件: ${file}`);
          } else if (stat.isDirectory()) {
            // 如果是目录，跳过（避免误删重要目录）
            console.log(`跳过目录: ${file}`);
          }
        }
      } catch (error) {
        console.warn(`删除文件失败: ${file}, 错误: ${error.message}`);
      }
    }
    
    // 更新SQLite数据库中的机器ID
    const dbUpdateResult = await ipcMain.handle('update-cursor-sqlite-db', null, dbPath);
    
    return { 
      success: true, 
      deletedFiles,
      dbPath,
      dbUpdateResult
    };
  } catch (error) {
    console.error('重置机器ID失败:', error);
    return { success: false, error: error.message };
  }
});

// 检查Cursor是否正在运行的内部函数
async function checkCursorRunning() {
  try {
    const { exec } = require('child_process');

    return new Promise((resolve) => {
      if (process.platform === 'win32') {
        exec('tasklist /FI "IMAGENAME eq Cursor.exe"', (error, stdout) => {
          if (error) {
            resolve(false);
            return;
          }
          const isRunning = stdout.includes('Cursor.exe');
          console.log(`Cursor运行状态: ${isRunning ? '运行中' : '未运行'}`);
          resolve(isRunning);
        });
      } else if (process.platform === 'darwin') {
        // 更精确地检测 Cursor 应用，排除系统的 CursorUIViewService
        exec('pgrep -f "Cursor.app" | grep -v CursorUIViewService', (error, stdout) => {
          if (error) {
            resolve(false);
            return;
          }
          const isRunning = stdout.trim().length > 0;
          console.log(`Cursor运行状态: ${isRunning ? '运行中' : '未运行'}`);
          resolve(isRunning);
        });
      } else {
        exec('pgrep -f cursor', (error, stdout) => {
          if (error) {
            resolve(false);
            return;
          }
          const isRunning = stdout.trim().length > 0;
          console.log(`Cursor运行状态: ${isRunning ? '运行中' : '未运行'}`);
          resolve(isRunning);
        });
      }
    });
  } catch (error) {
    console.error('检查Cursor运行状态失败:', error);
    return false;
  }
}

// 检查Cursor是否正在运行的IPC处理程序
ipcMain.handle('check-cursor-running', async () => {
  return await checkCursorRunning();
});

// 强制退出Cursor的内部函数
async function forceQuitCursor() {
  try {
    console.log('正在强制退出Cursor...');
    const { exec } = require('child_process');

    return new Promise((resolve) => {
      if (process.platform === 'win32') {
        // Windows: 使用taskkill强制终止
        exec('taskkill /F /IM Cursor.exe', (error, stdout, stderr) => {
          if (error) {
            console.log('没有找到运行中的Cursor进程或无法终止');
            resolve({ success: true, message: '没有运行中的Cursor进程' });
            return;
          }
          console.log('成功强制退出Cursor');
          resolve({ success: true, message: '成功强制退出Cursor' });
        });
      } else if (process.platform === 'darwin') {
        // macOS: 先尝试优雅退出，然后强制终止
        exec('osascript -e \'tell application "Cursor" to quit\'', (error) => {
          if (error) {
            console.log('优雅退出失败，尝试强制终止...');
            // 如果优雅退出失败，使用pkill强制终止
            exec('pkill -f "Cursor.app" | grep -v CursorUIViewService', (killError) => {
              if (killError) {
                console.log('没有找到运行中的Cursor进程');
                resolve({ success: true, message: '没有运行中的Cursor进程' });
              } else {
                console.log('成功强制退出Cursor');
                resolve({ success: true, message: '成功强制退出Cursor' });
              }
            });
          } else {
            console.log('成功优雅退出Cursor');
            resolve({ success: true, message: '成功优雅退出Cursor' });
          }
        });
      } else {
        // Linux: 使用pkill强制终止
        exec('pkill -f cursor', (error) => {
          if (error) {
            console.log('没有找到运行中的Cursor进程');
            resolve({ success: true, message: '没有运行中的Cursor进程' });
          } else {
            console.log('成功强制退出Cursor');
            resolve({ success: true, message: '成功强制退出Cursor' });
          }
        });
      }
    });
  } catch (error) {
    console.error('强制退出Cursor失败:', error);
    return { success: false, error: error.message };
  }
}

// 强制退出Cursor的IPC处理程序
ipcMain.handle('force-quit-cursor', async () => {
  return await forceQuitCursor();
});

// 启动Cursor
ipcMain.handle('launch-cursor', async () => {
  try {
    console.log('正在启动Cursor...');

    // 获取Cursor路径
    const pathsResult = await getCursorPaths();
    if (pathsResult.error) {
      throw new Error(pathsResult.error);
    }

    let executablePath;
    let cursorInstallPath;

    if (process.platform === 'win32') {
      // Windows: 从resources/app路径推导出安装路径
      cursorInstallPath = path.dirname(path.dirname(pathsResult.basePath)); // 去掉resources/app
      executablePath = path.join(cursorInstallPath, 'Cursor.exe');
    } else if (process.platform === 'darwin') {
      // macOS: 从Contents/Resources/app路径推导出.app路径
      cursorInstallPath = path.dirname(path.dirname(path.dirname(pathsResult.basePath))); // 去掉Contents/Resources/app
      executablePath = cursorInstallPath; // .app路径
    } else {
      // Linux: 从resources/app路径推导出安装路径
      cursorInstallPath = path.dirname(path.dirname(pathsResult.basePath));
      executablePath = path.join(cursorInstallPath, 'cursor');
    }

    console.log(`Cursor安装路径: ${cursorInstallPath}`);
    console.log(`Cursor可执行文件: ${executablePath}`);

    // 检查可执行文件是否存在
    if (!fs.existsSync(executablePath)) {
      throw new Error(`Cursor可执行文件不存在: ${executablePath}`);
    }

    const { spawn } = require('child_process');

    if (process.platform === 'win32') {
      // Windows
      const cursorProcess = spawn(executablePath, [], {
        detached: true,
        stdio: 'ignore'
      });
      cursorProcess.unref();
    } else if (process.platform === 'darwin') {
      // macOS - 使用open命令启动.app
      const cursorProcess = spawn('open', ['-a', executablePath], {
        detached: true,
        stdio: 'ignore'
      });
      cursorProcess.unref();
    } else {
      // Linux
      const cursorProcess = spawn(executablePath, [], {
        detached: true,
        stdio: 'ignore'
      });
      cursorProcess.unref();
    }

    console.log('Cursor启动成功');
    return { success: true, message: 'Cursor启动成功' };

  } catch (error) {
    console.error('启动Cursor失败:', error);
    return { success: false, error: error.message };
  }
});

// 探索Cursor配置结构函数已移除

// 检查Cursor数据库中的模型设置函数已移除

// 设置Cursor默认AI模型 (改进版)
ipcMain.handle('set-cursor-default-model', async (event, model = 'claude-3.5-sonnet') => {
  try {
    console.log(`正在设置Cursor默认AI模型为: ${model}`);

    const results = {
      settingsFile: { attempted: false, success: false, path: '', error: null },
      database: { attempted: false, success: false, path: '', error: null, updatedKeys: [] }
    };

    // 方法1: 尝试修改settings.json文件
    try {
      let settingsPath = '';
      if (process.platform === 'win32') {
        settingsPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'settings.json');
      } else if (process.platform === 'darwin') {
        settingsPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'settings.json');
      } else {
        settingsPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'settings.json');
      }

      results.settingsFile.attempted = true;
      results.settingsFile.path = settingsPath;

      console.log(`尝试修改设置文件: ${settingsPath}`);

      // 确保目录存在
      const settingsDir = path.dirname(settingsPath);
      if (!fs.existsSync(settingsDir)) {
        fs.mkdirSync(settingsDir, { recursive: true });
        console.log(`已创建设置目录: ${settingsDir}`);
      }

      // 读取现有设置
      let cursorSettings = {};
      if (fs.existsSync(settingsPath)) {
        try {
          const settingsContent = fs.readFileSync(settingsPath, 'utf8');
          cursorSettings = JSON.parse(settingsContent);
          console.log('已读取现有设置文件');
        } catch (parseError) {
          console.warn('解析现有设置失败，将创建新设置:', parseError.message);
          cursorSettings = {};
        }
      }

      // 设置默认AI模型的多种可能键名
      const modelKeys = [
        'cursor.chat.defaultModel',
        'cursor.general.defaultModel',
        'cursor.defaultModel',
        'chat.defaultModel',
        'ai.defaultModel',
        'workbench.chat.defaultModel'
      ];

      for (const key of modelKeys) {
        cursorSettings[key] = model;
      }

      // 创建备份
      if (fs.existsSync(settingsPath)) {
        const backupPath = `${settingsPath}.bak`;
        fs.copyFileSync(settingsPath, backupPath);
        console.log(`已创建设置文件备份: ${backupPath}`);
      }

      // 写入新设置
      fs.writeFileSync(settingsPath, JSON.stringify(cursorSettings, null, 2), 'utf8');
      console.log(`已成功修改设置文件，设置模型为: ${model}`);

      results.settingsFile.success = true;

    } catch (settingsError) {
      console.error('修改设置文件失败:', settingsError);
      results.settingsFile.error = settingsError.message;
    }

    // 方法2: 尝试修改数据库
    try {
      let dbPath = '';
      if (process.platform === 'win32') {
        dbPath = path.join(os.homedir(), 'AppData', 'Roaming', 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      } else if (process.platform === 'darwin') {
        dbPath = path.join(os.homedir(), 'Library', 'Application Support', 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      } else {
        dbPath = path.join(os.homedir(), '.config', 'Cursor', 'User', 'globalStorage', 'state.vscdb');
      }

      results.database.attempted = true;
      results.database.path = dbPath;

      if (fs.existsSync(dbPath)) {
        console.log(`尝试修改数据库: ${dbPath}`);

        // 使用新的数据库封装
        const db = new CursorDatabase(dbPath);
        const connectResult = db.connect();

        if (!connectResult.success) {
          results.database.success = false;
          results.database.error = connectResult.error;
        } else {
          // 检查表是否存在
          const tableCheck = db.checkTable('ItemTable');
          if (!tableCheck.success || (!tableCheck.exists && !tableCheck.simulation)) {
            db.close();
            results.database.success = false;
            results.database.error = 'ItemTable表不存在';
          } else {
            // 尝试设置多种可能的模型键
            const modelKeys = [
              'cursor.chat.defaultModel',
              'cursor.general.defaultModel',
              'cursor.defaultModel',
              'chat.defaultModel',
              'ai.defaultModel',
              'workbench.chat.defaultModel',
              'vscode.chat.defaultModel'
            ];

            // 准备批量更新数据
            const updates = modelKeys.map(key => [key, model]);

            // 执行批量更新
            const updateResult = db.batchUpdate(updates);

            db.close();

            if (updateResult.success) {
              results.database.success = true;
              results.database.updatedKeys = updateResult.updatedKeys || modelKeys;
              results.database.message = `成功更新 ${updateResult.updatedKeys?.length || modelKeys.length} 个模型配置键`;
            } else {
              results.database.success = false;
              results.database.error = updateResult.error;
            }
          }
        }
      } else {
        results.database.error = '数据库文件不存在';
      }

    } catch (dbError) {
      console.error('修改数据库失败:', dbError);
      results.database.error = dbError.message;
    }

    // 判断整体成功状态
    const overallSuccess = results.settingsFile.success || results.database.success;

    return {
      success: overallSuccess,
      message: overallSuccess ? `已成功设置默认AI模型为: ${model}` : '设置默认模型失败',
      model,
      details: results
    };

  } catch (error) {
    console.error('设置默认AI模型失败:', error);
    return { success: false, error: error.message };
  }
});

// 重启Cursor
ipcMain.handle('restart-cursor', async (event, cursorPath) => {
  try {
    // 获取Cursor可执行文件路径
    let executablePath;

    if (process.platform === 'win32') {
      executablePath = path.join(cursorPath, 'Cursor.exe');

      // 先关闭现有的Cursor进程
      const { exec } = require('child_process');
      exec('taskkill /F /IM Cursor.exe', (error) => {
        if (error) {
          console.log('没有运行中的Cursor进程或无法关闭进程');
        } else {
          console.log('成功关闭Cursor进程');
        }

        // 延迟1秒后启动新进程
        setTimeout(() => {
          // 启动Cursor进程
          const { spawn } = require('child_process');
          const cursorProcess = spawn(executablePath, [], {
            detached: true,
            stdio: 'ignore'
          });

          // 分离进程
          cursorProcess.unref();
          console.log(`Cursor已重启: ${executablePath}`);
        }, 1000);
      });
    } else if (process.platform === 'darwin') {
      executablePath = path.join(cursorPath, 'Contents', 'MacOS', 'Cursor');

      // 先关闭现有的Cursor进程
      const { exec } = require('child_process');
      exec('pkill -9 Cursor', (error) => {
        if (error) {
          console.log('没有运行中的Cursor进程或无法关闭进程');
        } else {
          console.log('成功关闭Cursor进程');
        }

        // 延迟1秒后启动新进程
        setTimeout(() => {
          // 启动Cursor进程
          const { spawn } = require('child_process');
          const cursorProcess = spawn('open', ['-a', executablePath], {
            detached: true,
            stdio: 'ignore'
          });

          // 分离进程
          cursorProcess.unref();
          console.log(`Cursor已重启: ${executablePath}`);
        }, 1000);
      });
    } else {
      executablePath = path.join(cursorPath, 'cursor');

      // 先关闭现有的Cursor进程
      const { exec } = require('child_process');
      exec('pkill -9 Cursor', (error) => {
        if (error) {
          console.log('没有运行中的Cursor进程或无法关闭进程');
        } else {
          console.log('成功关闭Cursor进程');
        }

        // 延迟1秒后启动新进程
        setTimeout(() => {
          // 启动Cursor进程
          const { spawn } = require('child_process');
          const cursorProcess = spawn(executablePath, [], {
            detached: true,
            stdio: 'ignore'
          });

          // 分离进程
          cursorProcess.unref();
          console.log(`Cursor已重启: ${executablePath}`);
        }, 1000);
      });
    }

    return true;
  } catch (error) {
    console.error(`重启Cursor失败: ${error.message}`);
    return false;
  }
});

// 热切换账号功能（Cursor运行时）
ipcMain.handle('hot-switch-account', async (event, dbPath, email, access_token, refresh_token) => {
  try {
    console.log('=== 热切换账号开始 ===');
    console.log(`数据库路径: ${dbPath}`);
    console.log(`用户邮箱: ${email}`);
    console.log(`access_token长度: ${access_token ? access_token.length : 0}`);
    console.log(`refresh_token长度: ${refresh_token ? refresh_token.length : 0}`);

    // 参数验证
    if (!dbPath) {
      return { success: false, error: '数据库路径不能为空' };
    }

    if (!email) {
      return { success: false, error: '邮箱不能为空' };
    }

    if (!access_token) {
      return { success: false, error: 'access_token不能为空' };
    }

    if (!refresh_token) {
      return { success: false, error: 'refresh_token不能为空' };
    }

    if (!fs.existsSync(dbPath)) {
      return { success: false, error: `数据库文件不存在: ${dbPath}` };
    }

    // 数据库功能现在通过 CursorDatabase 封装提供

    // 创建数据库备份
    const backupPath = `${dbPath}.bak`;
    if (!fs.existsSync(backupPath)) {
      try {
        fs.copyFileSync(dbPath, backupPath);
        console.log(`已创建数据库备份: ${backupPath}`);
      } catch (backupError) {
        console.warn(`创建备份失败，但将继续执行: ${backupError.message}`);
      }
    }

    // 热切换：只更新数据库中的账号信息，不重置机器ID
    console.log('正在连接数据库...');

    // 使用新的数据库封装
    const db = new CursorDatabase(dbPath);
    const connectResult = db.connect();

    if (!connectResult.success) {
      return connectResult;
    }

    console.log('已成功连接到数据库');

    // 使用数据库封装的更新认证信息方法
    const updateResult = db.updateCursorAuth(email, access_token, refresh_token);

    db.close();

    const dbResult = updateResult;



    if (!dbResult.success) {
      return dbResult;
    }

    console.log('热切换账号完成！');
    console.log(`更新的键: ${dbResult.updatedKeys?.join(', ') || '无'}`);

    return {
      success: true,
      message: '账号热切换成功',
      updatedKeys: dbResult.updatedKeys,
      email: email
    };

  } catch (error) {
    console.error('热切换账号失败:', error);
    return { success: false, error: error.message };
  }
});

// 通知Cursor重新加载配置
ipcMain.handle('notify-cursor-reload', async () => {
  try {
    console.log('正在通知Cursor重新加载配置...');

    // 方法1: 尝试发送系统通知给Cursor进程
    if (process.platform === 'darwin') {
      // macOS: 使用AppleScript发送重新加载信号
      const { exec } = require('child_process');

      return new Promise((resolve) => {
        // 尝试通过AppleScript告诉Cursor重新加载
        const script = `
          tell application "System Events"
            try
              tell process "Cursor"
                -- 发送 Cmd+Shift+P 打开命令面板
                key code 35 using {command down, shift down}
                delay 0.5
                -- 输入重新加载命令
                keystroke "Developer: Reload Window"
                delay 0.5
                -- 按回车执行
                key code 36
              end tell
              return "success"
            on error
              return "error"
            end try
          end tell
        `;

        exec(`osascript -e '${script}'`, (error, stdout) => {
          if (error) {
            console.log('AppleScript方法失败，使用备用方案');
            resolve({ success: false, method: 'applescript', error: error.message });
          } else {
            console.log('AppleScript通知成功');
            resolve({ success: true, method: 'applescript' });
          }
        });
      });
    } else {
      // Windows/Linux: 暂时返回成功，用户需要手动重新加载
      console.log('非macOS系统，建议用户手动重新加载Cursor');
      return { success: true, method: 'manual', message: '请在Cursor中按Ctrl+Shift+P，然后输入"Developer: Reload Window"来重新加载' };
    }

  } catch (error) {
    console.error('通知Cursor重新加载失败:', error);
    return { success: false, error: error.message };
  }
});

// 基于Python项目逻辑的账号切换功能
ipcMain.handle('python-style-account-switch', async (event, dbPath, email, access_token, refresh_token, resetDeviceId = true) => {
  try {
    console.log('=== Python风格账号切换开始 ===');
    console.log(`数据库路径: ${dbPath}`);
    console.log(`用户邮箱: ${email}`);
    console.log(`access_token长度: ${access_token ? access_token.length : 0}`);
    console.log(`refresh_token长度: ${refresh_token ? refresh_token.length : 0}`);
    console.log(`是否重置设备ID: ${resetDeviceId}`);

    // 参数验证
    if (!dbPath) {
      return { success: false, error: '数据库路径不能为空' };
    }

    if (!email) {
      return { success: false, error: '邮箱不能为空' };
    }

    if (!access_token) {
      return { success: false, error: 'access_token不能为空' };
    }

    if (!refresh_token) {
      return { success: false, error: 'refresh_token不能为空' };
    }

    // 步骤1: 检查并强制退出Cursor
    console.log('步骤1: 检查Cursor运行状态...');

    // 直接调用检查函数而不是通过IPC
    const isRunning = await checkCursorRunning();

    if (isRunning) {
      console.log('检测到Cursor正在运行，正在强制退出...');

      // 直接调用强制退出函数而不是通过IPC
      const quitResult = await forceQuitCursor();

      if (!quitResult.success) {
        console.error('强制退出Cursor失败:', quitResult.error);
        return { success: false, error: `强制退出Cursor失败: ${quitResult.error}` };
      }

      console.log('Cursor已成功退出，等待3秒确保进程完全结束...');
      await new Promise(resolve => setTimeout(resolve, 3000));
    } else {
      console.log('Cursor未运行，可以直接进行账号切换');
    }

    if (!fs.existsSync(dbPath)) {
      return { success: false, error: `数据库文件不存在: ${dbPath}` };
    }

    // 数据库功能现在通过 CursorDatabase 封装提供，支持模拟模式

    // 创建数据库备份
    const backupPath = `${dbPath}.bak`;
    if (!fs.existsSync(backupPath)) {
      try {
        fs.copyFileSync(dbPath, backupPath);
        console.log(`已创建数据库备份: ${backupPath}`);
      } catch (backupError) {
        console.warn(`创建备份失败，但将继续执行: ${backupError.message}`);
      }
    }

    // 重置机器ID（可选：根据参数决定是否重置机器码）
    if (resetDeviceId) {
      try {
        console.log('开始重置机器ID...');

      // 获取正确的Cursor目录路径（支持自定义路径）
      let cursorDir;

      // 首先尝试从设置中获取自定义路径
      try {
        const settings = loadSettings();
        if (settings && settings.customCursorPath && fs.existsSync(settings.customCursorPath)) {
          cursorDir = settings.customCursorPath;
          console.log(`使用自定义Cursor路径: ${cursorDir}`);
        } else {
          // 使用默认路径
          if (process.platform === 'win32') {
            cursorDir = path.join(process.env.APPDATA, 'Cursor');
          } else if (process.platform === 'darwin') {
            cursorDir = path.join(process.env.HOME, 'Library', 'Application Support', 'Cursor');
          } else {
            cursorDir = path.join(process.env.HOME, '.config', 'Cursor');
          }
          console.log(`使用默认Cursor路径: ${cursorDir}`);
        }
      } catch (settingsError) {
        console.warn(`读取设置失败，使用默认路径: ${settingsError.message}`);
        // 使用默认路径
        if (process.platform === 'win32') {
          cursorDir = path.join(process.env.APPDATA, 'Cursor');
        } else if (process.platform === 'darwin') {
          cursorDir = path.join(process.env.HOME, 'Library', 'Application Support', 'Cursor');
        } else {
          cursorDir = path.join(process.env.HOME, '.config', 'Cursor');
        }
        console.log(`使用默认Cursor路径: ${cursorDir}`);
      }

      console.log(`Cursor目录路径: ${cursorDir}`);
      
      // 针对Mac系统使用专门的storage.json重置方法
      if (process.platform === 'darwin') {
        console.log('🍎 Mac系统：执行专门的storage.json设备ID重置...');
        const macResetResult = await resetMacStorageJsonDeviceIds(cursorDir);
        
        if (macResetResult.success) {
          console.log('✅ Mac storage.json设备ID重置成功');
          console.log('更新字段:', macResetResult.updatedFields?.join(', '));
          console.log('新的设备ID:', {
            macMachineId: macResetResult.newIds?.macMachineId?.substring(0, 12) + '...',
            machineId: macResetResult.newIds?.machineId?.substring(0, 12) + '...',
            devDeviceId: macResetResult.newIds?.devDeviceId
          });
        } else {
          console.warn('⚠️ Mac storage.json重置失败，使用通用重置方法:', macResetResult.error);
        }
      }
      
        // 执行通用的机器ID重置（作为补充或备选）
        await resetMachineIds(cursorDir);
        console.log('机器ID重置完成');
      } catch (resetError) {
        console.warn(`重置机器ID过程中出错: ${resetError.message}`);
        // 不中断流程，继续执行
      }
    } else {
      console.log('跳过设备ID重置（用户选择保持现有设备ID）');
    }

    // 检查是否需要使用模拟模式（通过 CursorDatabase 自动处理）
    const db = new CursorDatabase(dbPath);
    const connectResult = db.connect();

    if (connectResult.simulation) {
      console.log('使用模拟数据库操作模式');

      // 模拟数据库操作
      const simulatedUpdates = [
        'cursorAuth/cachedEmail',
        'cursorAuth/accessToken',
        'cursorAuth/refreshToken',
        'cursorAuth/cachedSignUpType'
      ];

      console.log('模拟删除cursorai/serverConfig以激活新账号...');
      console.log('模拟更新认证信息...');

      // 模拟延迟
      setTimeout(() => {
        console.log('=== Python风格账号切换完成（模拟模式）===');
        console.log(`模拟更新的键: ${simulatedUpdates.join(', ')}`);
      }, 100);

      return Promise.resolve({
        success: true,
        message: 'Python风格账号切换成功（模拟模式）',
        updatedKeys: simulatedUpdates,
        mode: 'simulated'
      });
    }

    // 重用之前创建的数据库连接
    console.log('正在连接数据库...');
    // db 变量已在上面创建，这里重新连接
    db.close(); // 先关闭之前的连接
    const connectResult2 = db.connect();

    if (!connectResult2.success) {
      return connectResult2;
    }

    console.log('已成功连接到数据库');

    // 检查表是否存在
    const tableCheck = db.checkTable('ItemTable');
    if (!tableCheck.success || (!tableCheck.exists && !tableCheck.simulation)) {
      db.close();
      return { success: false, error: 'ItemTable表不存在' };
    }

    // 使用数据库封装的更新认证信息方法
    const updateResult = db.updateCursorAuth(email, access_token, refresh_token);

    db.close();

    if (updateResult.success) {
      return {
        success: true,
        message: '账号切换成功',
        updatedKeys: updateResult.updatedKeys || ['cursorAuth/cachedEmail', 'cursorAuth/accessToken', 'cursorAuth/refreshToken', 'cursorAuth/cachedSignUpType']
      };
    } else {
      return {
        success: false,
        error: `账号切换失败: ${updateResult.error}`
      };
    }


  } catch (error) {
    console.error('Python风格账号切换失败:', error);
    return { success: false, error: error.message };
  }
});

// 更新Cursor的认证信息
ipcMain.handle('update-cursor-auth', async (event, dbPath, email, access_token, refresh_token, machineIdReset = true) => {
  try {
    if (!fs.existsSync(dbPath)) {
      return { success: false, error: '数据库文件不存在' };
    }
    
    console.log('正在更新Cursor认证信息...');
    console.log(`数据库路径: ${dbPath}`);
    console.log(`用户邮箱: ${email}`);
    
    // 数据库功能现在通过 CursorDatabase 封装提供
    
    // 创建数据库备份
    const backupPath = `${dbPath}.bak`;
    if (!fs.existsSync(backupPath)) {
      try {
        fs.copyFileSync(dbPath, backupPath);
        console.log(`已创建数据库备份: ${backupPath}`);
      } catch (backupError) {
        console.warn(`创建备份失败，但将继续执行: ${backupError.message}`);
        // 不中断流程，继续执行
      }
    }
    
    // 重置机器ID部分
    if (machineIdReset) {
      try {
        console.log('开始重置机器ID...');
        await resetMachineIds(path.dirname(path.dirname(dbPath)));
      } catch (resetError) {
        console.warn(`重置机器ID过程中出错: ${resetError.message}`);
        // 不中断流程，继续执行
      }
    }
    
    // 使用新的数据库封装
    const db = new CursorDatabase(dbPath);
    const connectResult = db.connect();

    if (!connectResult.success) {
      return connectResult;
    }

    console.log('已连接到数据库');

    // 检查表是否存在
    const tableCheck = db.checkTable('ItemTable');
    if (!tableCheck.success || (!tableCheck.exists && !tableCheck.simulation)) {
      db.close();
      return { success: false, error: 'ItemTable表不存在' };
    }

    // 使用数据库封装的更新认证信息方法
    const updateResult = db.updateCursorAuth(email, access_token, refresh_token);

    db.close();

    if (updateResult.success) {
      return {
        success: true,
        message: '认证信息更新成功',
        updatedKeys: updateResult.updatedKeys || ['cursorAuth/cachedEmail', 'cursorAuth/accessToken', 'cursorAuth/refreshToken', 'cursorAuth/cachedSignUpType']
      };
    } else {
      return {
        success: false,
        error: `认证信息更新失败: ${updateResult.error}`
      };
    }
  } catch (error) {
    console.error('更新认证信息失败:', error);
    return { success: false, error: error.message };
  }
});

// Mac专用的storage.json设备ID重置函数 - 借鉴Python版本完整实现
async function resetMacStorageJsonDeviceIds(cursorDir) {
  const crypto = require('crypto');
  const { v4: uuidv4 } = require('uuid');
  
  try {
    console.log('🍎 开始Mac专用设备ID重置 - 借鉴Python版本完整算法...');
    
    // 借鉴Python版本的完整ID生成策略 (reset_machine_manual.py 第563-585行)
    // 1. 设备UUID (标准格式) - 对应 telemetry.devDeviceId
    const newDevDeviceId = uuidv4();
    
    // 2. 机器ID (SHA256, 64位十六进制) - 对应 telemetry.machineId  
    const newMachineId = crypto.createHash('sha256').update(crypto.randomBytes(32)).digest('hex');
    
    // 3. Mac机器ID (SHA512, 128位十六进制) - 对应 telemetry.macMachineId
    const newMacMachineId = crypto.createHash('sha512').update(crypto.randomBytes(64)).digest('hex');
    
    // 4. 系统质量监控ID (大写UUID格式) - 对应 telemetry.sqmId
    const newSqmId = '{' + uuidv4().toUpperCase() + '}';
    
    // 5. 服务机器ID (与devDeviceId相同) - 对应 storage.serviceMachineId
    const newServiceMachineId = newDevDeviceId;
    
    // 完整的ID映射，与Python版本保持一致
    const newIds = {
      'telemetry.devDeviceId': newDevDeviceId,
      'telemetry.macMachineId': newMacMachineId,
      'telemetry.machineId': newMachineId,
      'telemetry.sqmId': newSqmId,
      'storage.serviceMachineId': newServiceMachineId
    };
    
    console.log(`📋 为Mac生成完整的设备ID集合:`);
    console.log(`  🔷 telemetry.devDeviceId: ${newDevDeviceId}`);
    console.log(`  🔷 telemetry.machineId: ${newMachineId.substring(0, 16)}... (${newMachineId.length}位)`);
    console.log(`  🔷 telemetry.macMachineId: ${newMacMachineId.substring(0, 16)}... (${newMacMachineId.length}位)`);
    console.log(`  🔷 telemetry.sqmId: ${newSqmId}`);
    console.log(`  🔷 storage.serviceMachineId: ${newServiceMachineId}`);
    
    // storage.json文件路径
    const storageJsonPath = path.join(cursorDir, 'User', 'globalStorage', 'storage.json');
    console.log(`📁 Mac storage.json路径: ${storageJsonPath}`);
    
    let storageData = {};
    let fileExists = false;
    
    // 检查文件是否存在并读取
    if (fs.existsSync(storageJsonPath)) {
      fileExists = true;
      try {
        const storageContent = fs.readFileSync(storageJsonPath, 'utf8');
        storageData = JSON.parse(storageContent);
        console.log('📖 已读取现有的storage.json文件');
      } catch (parseError) {
        console.warn('⚠️ 解析storage.json失败，将创建新文件:', parseError.message);
        storageData = {};
      }
    } else {
      console.log('📝 storage.json文件不存在，将创建新文件');
      // 确保目录存在
      const storageDir = path.dirname(storageJsonPath);
      if (!fs.existsSync(storageDir)) {
        fs.mkdirSync(storageDir, { recursive: true });
        console.log(`📂 已创建目录: ${storageDir}`);
      }
    }
    
    // 创建备份 - 借鉴Python版本的备份策略
    if (fileExists) {
      try {
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace(/T/, '_').split('.')[0];
        const backupPath = `${storageJsonPath}.bak.${timestamp}`;
        fs.copyFileSync(storageJsonPath, backupPath);
        console.log(`💾 已创建storage.json备份: ${path.basename(backupPath)}`);
      } catch (backupError) {
        console.warn('⚠️ 创建备份失败:', backupError.message);
      }
    }
    
    // 借鉴Python版本，批量更新所有关键的设备ID字段
    console.log('🔄 正在更新storage.json中的设备ID字段...');
    let updatedCount = 0;
    for (const [key, value] of Object.entries(newIds)) {
      const oldValue = storageData[key];
      storageData[key] = value;
      updatedCount++;
      
      if (oldValue) {
        console.log(`  ✏️ 更新 ${key}: ${oldValue.substring(0, 8)}... → ${value.substring(0, 8)}...`);
      } else {
        console.log(`  ➕ 新增 ${key}: ${value.substring(0, 8)}...`);
      }
    }
    
    // 原子性写入文件
    try {
      fs.writeFileSync(storageJsonPath, JSON.stringify(storageData, null, 2), 'utf8');
      console.log(`✅ Mac storage.json设备ID重置成功，更新了 ${updatedCount} 个字段`);
    } catch (writeError) {
      console.error('❌ 写入storage.json失败:', writeError);
      throw writeError;
    }

    // 启动设备ID保护监控，防止Cursor重启时修改回原值
    console.log('🛡️ 启动设备ID持续保护监控...');
    const protectionStarted = startDeviceIdProtection(storageJsonPath, newIds);
    if (protectionStarted) {
      console.log('✅ 设备ID保护监控已激活，将持续防护直到应用关闭');
    } else {
      console.warn('⚠️ 设备ID保护监控启动失败，重启Cursor时可能会恢复原值');
    }

    return {
      success: true,
      updatedFields: Object.keys(newIds),
      updatedCount: updatedCount,
      newIds: {
        devDeviceId: newDevDeviceId,
        machineId: newMachineId,
        macMachineId: newMacMachineId,
        sqmId: newSqmId,
        serviceMachineId: newServiceMachineId
      },
      // 返回完整的ID映射，供SQLite更新使用
      completeIdMap: newIds,
      protectionActive: protectionStarted
    };
    
  } catch (error) {
    console.error('❌ Mac storage.json设备ID重置失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

// 添加重置机器ID的函数 - 借鉴Python版本的完整实现
async function resetMachineIds(cursorDir) {
  const crypto = require('crypto');
  const { v4: uuidv4 } = require('uuid');
  
  // 借鉴Python版本的ID生成算法
  const newIds = {
    devDeviceId: uuidv4(),
    machineId: crypto.createHash('sha256').update(crypto.randomBytes(32)).digest('hex'),
    macMachineId: crypto.createHash('sha512').update(crypto.randomBytes(64)).digest('hex'),
    sqmId: '{' + uuidv4().toUpperCase() + '}',
    serviceMachineId: uuidv4()
  };
  
  console.log(`生成新的设备ID: ${newIds.devDeviceId}`);
  console.log(`生成新的机器ID: ${newIds.machineId.substring(0, 16)}...`);
  console.log(`生成新的Mac机器ID: ${newIds.macMachineId.substring(0, 16)}...`);
  console.log(`生成新的SQM ID: ${newIds.sqmId}`);
  console.log(`生成新的服务机器ID: ${newIds.serviceMachineId}`);
  
  // 对于Mac系统，首先执行专门的storage.json重置
  if (process.platform === 'darwin') {
    console.log('🍎 检测到Mac系统，执行专门的storage.json设备ID重置...');
    const macResetResult = await resetMacStorageJsonDeviceIds(cursorDir);
    
    if (macResetResult.success) {
      console.log('✅ Mac storage.json设备ID重置成功');
      console.log('更新的字段:', macResetResult.updatedFields.join(', '));
    } else {
      console.warn('⚠️ Mac storage.json设备ID重置失败:', macResetResult.error);
      // 继续执行通用重置逻辑作为备选
    }
  }
  
  // 更新SQLite数据库 - 借鉴Python版本的实现，传递完整的ID映射
  const sqliteResult = await updateSQLiteDatabase(cursorDir, { completeIdMap: newIds });
  if (sqliteResult.success) {
    console.log(`✅ SQLite数据库更新成功: ${sqliteResult.updatedCount} 个字段`);
  } else {
    console.warn(`⚠️ SQLite数据库更新部分失败: ${sqliteResult.message}`);
  }
  
  // macOS系统级集成 - 更新Platform UUID (借鉴Python版本)
  if (process.platform === 'darwin') {
    try {
      console.log('🍎 开始macOS系统级Platform UUID更新...');
      const systemResult = await updateMacOSPlatformUUID(newIds);
      if (systemResult.success) {
        console.log('✅ macOS系统级UUID更新成功');
      } else {
        console.warn('⚠️ macOS系统级UUID更新失败:', systemResult.error);
      }
    } catch (systemError) {
      console.warn('⚠️ macOS系统级集成过程中出错:', systemError.message);
    }
  }
  
  // 更新独立的machineId文件 - 借鉴Python版本实现
  try {
    console.log('📄 开始更新独立machineId文件...');
    const deviceId = newIds['telemetry.devDeviceId'] || newIds.devDeviceId;
    const machineIdResult = await updateMachineIdFile(cursorDir, deviceId);
    if (machineIdResult.success) {
      console.log('✅ 独立machineId文件更新成功');
    } else {
      console.warn('⚠️ 独立machineId文件更新失败:', machineIdResult.error);
    }
  } catch (machineIdError) {
    console.warn('⚠️ 独立machineId文件处理过程中出错:', machineIdError.message);
  }
  
  // 应用程序文件修改 - 借鉴Python版本的完整修改 (reset_machine_manual.py 第754-766行)
  try {
    console.log('🔧 开始应用程序文件修改 - 借鉴Python版本完整实现...');
    
    // 获取Cursor应用路径信息
    const pathsResult = await getCursorPaths();
    if (pathsResult && !pathsResult.error) {
      console.log(`📂 找到Cursor应用路径: ${pathsResult.basePath}`);
      console.log(`📋 Cursor版本: ${pathsResult.version || '未知版本'}`);
      
      // 修改workbench.desktop.main.js - 借鉴Python版本
      try {
        console.log('🎨 修改workbench.desktop.main.js...');
        const workbenchResult = await modifyWorkbenchJS(pathsResult);
        if (workbenchResult.success) {
          console.log('✅ workbench.js修改成功');
        } else {
          console.warn('⚠️ workbench.js修改失败:', workbenchResult.error);
        }
      } catch (workbenchError) {
        console.warn('⚠️ workbench.js修改过程中出错:', workbenchError.message);
      }
      
      // 检查版本并修改main.js - 借鉴Python版本的版本检测逻辑
      try {
        const version = pathsResult.version || '0.0.0';
        const versionParts = version.split('.').map(Number);
        const currentVersion = versionParts[0] * 10000 + versionParts[1] * 100 + versionParts[2];
        const minVersion = 0 * 10000 + 45 * 100 + 0; // 0.45.0
        
        if (currentVersion >= minVersion) {
          console.log(`🔄 检测到Cursor版本 >= 0.45.0，执行高级补丁...`);
          const mainResult = await modifyMainJS(pathsResult);
          if (mainResult.success) {
            console.log('✅ main.js高级补丁应用成功');
          } else {
            console.warn('⚠️ main.js高级补丁应用失败:', mainResult.error);
          }
        } else {
          console.log(`📋 检测到较低版本 (${version})，跳过main.js高级补丁`);
        }
      } catch (versionError) {
        console.warn('⚠️ 版本检测失败，跳过main.js修改:', versionError.message);
      }
      
    } else {
      console.warn('⚠️ 无法获取Cursor应用路径，跳过应用程序文件修改');
      console.warn('错误:', pathsResult?.error || '未知错误');
    }
    
  } catch (appModifyError) {
    console.warn('⚠️ 应用程序文件修改过程中出错:', appModifyError.message);
    // 不中断流程，这个功能失败不影响机器码重置的核心功能
  }
  
  // 权限预检查
  try {
    // 检查目录是否存在且可访问
    if (!fs.existsSync(cursorDir)) {
      throw new Error(`Cursor目录不存在: ${cursorDir}`);
    }
    
    // 检查目录权限
    try {
      fs.accessSync(cursorDir, fs.constants.R_OK | fs.constants.W_OK);
    } catch (accessError) {
      console.warn(`目录权限检查失败: ${cursorDir}`);
      console.warn('这可能需要管理员权限或修改文件权限');
      // 不直接抛出错误，让具体的文件操作来处理
    }
    
    // 创建权限测试文件
    const testFile = path.join(cursorDir, '.permission-test');
    try {
      fs.writeFileSync(testFile, 'test');
      fs.unlinkSync(testFile);
      console.log('权限检查通过：可以创建和删除文件');
    } catch (permError) {
      console.warn(`写入权限测试失败: ${permError.message}`);
      console.warn('可能需要以下解决方案：');
      console.warn('1. 确保Cursor已完全退出');
      console.warn('2. 检查文件权限：chmod 755 "' + cursorDir + '"');
      console.warn('3. 在系统偏好设置中授予应用完整磁盘访问权限');
      // 继续执行，让具体操作来决定是否失败
    }
    
  } catch (preCheckError) {
    console.error(`权限预检查失败: ${preCheckError.message}`);
    throw preCheckError;
  }
  
  try {
    // 定义所有需要重置的机器ID文件
    let machineIdFiles = [];
    
    if (process.platform === 'darwin') {
      // macOS 特定的文件列表 - 基于实际目录结构
      machineIdFiles = [
        // 根目录下的机器ID文件
        path.join(cursorDir, 'machineid.json'),
        path.join(cursorDir, 'machineid'),
        // globalStorage 目录下的文件
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'anonymousid'),
        // 可能的其他位置
        path.join(cursorDir, 'User', 'workspaceStorage', 'machine-id'),
        path.join(cursorDir, 'logs', 'machine-id')
      ];
    } else if (process.platform === 'win32') {
      // Windows 文件列表
      machineIdFiles = [
        path.join(cursorDir, 'machineid.json'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'anonymousid')
      ];
    } else {
      // Linux 文件列表
      machineIdFiles = [
        path.join(cursorDir, 'machineid.json'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'anonymousid')
      ];
    }
    
    let processedFiles = 0;
    let errorFiles = [];
    
    for (const filePath of machineIdFiles) {
      try {
        // 确保目录存在
        const dir = path.dirname(filePath);
        if (!fs.existsSync(dir)) {
          fs.mkdirSync(dir, { recursive: true });
          console.log(`创建目录: ${dir}`);
        }

        const fileExtension = path.extname(filePath);
        const fileExists = fs.existsSync(filePath);
        const fileName = path.basename(filePath);

        // 检查文件权限（如果文件存在）
        if (fileExists) {
          try {
            fs.accessSync(filePath, fs.constants.R_OK | fs.constants.W_OK);
          } catch (accessError) {
            console.warn(`文件权限不足: ${filePath}`);
            console.warn(`权限错误: ${accessError.message}`);
            
            // 提供权限修复建议
            if (accessError.code === 'EACCES') {
              console.warn('建议解决方案：');
              console.warn(`  chmod 644 "${filePath}"`);
              console.warn('  或者以管理员权限运行应用');
            }
            
            throw accessError;
          }
        }

        if (fileExtension === '.json') {
          // JSON格式文件
          const data = { machineId: deviceId };
          fs.writeFileSync(filePath, JSON.stringify(data, null, 2));
          console.log(`${fileExists ? '更新' : '创建'}JSON机器ID文件: ${filePath}`);
          processedFiles++;
        } else {
          // 纯文本文件
          const idToWrite = fileName === 'anonymousid' ? anonymousId : deviceId;
          fs.writeFileSync(filePath, idToWrite);
          console.log(`${fileExists ? '更新' : '创建'}文本机器ID文件: ${filePath}`);
          processedFiles++;
        }
      } catch (e) {
        console.warn(`处理文件失败: ${filePath}`);
        console.warn(`错误详情: ${e.message}`);
        
        // 根据错误类型提供具体建议
        if (e.code === 'EACCES') {
          console.warn('❌ 权限被拒绝 - 这通常表示需要管理员权限');
          console.warn('💡 解决方案：');
          console.warn('   1. 确保Cursor应用已完全退出');
          console.warn('   2. 在系统偏好设置中授予应用完整磁盘访问权限');
          console.warn(`   3. 手动修复权限: chmod 644 "${filePath}"`);
        } else if (e.code === 'ENOENT') {
          console.warn('📁 目录不存在 - 将尝试创建');
        } else if (e.code === 'EBUSY' || e.code === 'EMFILE') {
          console.warn('🔒 文件被占用 - Cursor可能仍在运行');
          console.warn('💡 请完全退出Cursor后重试');
        } else {
          console.warn(`❓ 未知错误: ${e.code || '无错误代码'}`);
        }
        
        errorFiles.push({
          path: filePath,
          error: e.message,
          code: e.code,
          needsPermission: e.code === 'EACCES'
        });
      }
    }
    
    // macOS 特定处理：清除可能缓存的设备信息
    if (process.platform === 'darwin') {
      try {
        // 清除 Cursor 的缓存目录中可能存在的设备信息
        const cacheDir = path.join(cursorDir, 'Cache');
        if (fs.existsSync(cacheDir)) {
          console.log('正在清除 macOS 缓存目录中的设备信息...');
          // 不完全删除 Cache 目录，只清除可能包含设备ID的文件
          const cacheFiles = fs.readdirSync(cacheDir);
          for (const file of cacheFiles) {
            if (file.includes('machine') || file.includes('device') || file.includes('uuid')) {
              const cacheFilePath = path.join(cacheDir, file);
              try {
                if (fs.statSync(cacheFilePath).isFile()) {
                  fs.unlinkSync(cacheFilePath);
                  console.log(`删除缓存文件: ${cacheFilePath}`);
                }
              } catch (err) {
                console.warn(`删除缓存文件失败: ${cacheFilePath}, 错误: ${err.message}`);
              }
            }
          }
        }

        // 清除 SharedStorage 中可能的设备信息
        const sharedStoragePath = path.join(cursorDir, 'SharedStorage');
        if (fs.existsSync(sharedStoragePath)) {
          try {
            fs.unlinkSync(sharedStoragePath);
            console.log('已清除 SharedStorage 文件');
          } catch (err) {
            console.warn(`清除 SharedStorage 失败: ${err.message}`);
          }
        }

        // 清除 .sock 文件（可能包含会话信息）
        const sockFiles = fs.readdirSync(cursorDir).filter(file => file.endsWith('.sock'));
        for (const sockFile of sockFiles) {
          try {
            const sockPath = path.join(cursorDir, sockFile);
            fs.unlinkSync(sockPath);
            console.log(`删除 socket 文件: ${sockPath}`);
          } catch (err) {
            console.warn(`删除 socket 文件失败: ${sockFile}, 错误: ${err.message}`);
          }
        }

      } catch (macError) {
        console.warn(`macOS 特定清理过程中出错: ${macError.message}`);
      }
    }
    
    // 更新Windows注册表中的MachineGUID (仅在Windows上)
    if (process.platform === 'win32') {
      try {
        const { exec } = require('child_process');
        const regCommand = `reg add "HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\Cryptography" /v MachineGuid /t REG_SZ /d "${machineGuid}" /f`;
        
        exec(regCommand, (error, stdout, stderr) => {
          if (error) {
            console.warn(`更新Windows注册表失败: ${error.message}`);
          } else {
            console.log('成功更新Windows注册表MachineGUID');
          }
        });
      } catch (regError) {
        console.warn(`尝试更新注册表时出错: ${regError.message}`);
      }
    }
    
    // 报告处理结果
    console.log(`机器ID重置完成: 成功处理 ${processedFiles} 个文件`);
    
    if (errorFiles.length > 0) {
      console.warn(`处理失败的文件数量: ${errorFiles.length}`);
      
      // 统计权限错误
      const permissionErrors = errorFiles.filter(ef => ef.needsPermission);
      const otherErrors = errorFiles.filter(ef => !ef.needsPermission);
      
      if (permissionErrors.length > 0) {
        console.warn('\n❌ 权限错误的文件:');
        permissionErrors.forEach(ef => {
          console.warn(`   ${ef.path}: ${ef.error}`);
        });
        console.warn('\n🔧 权限问题解决建议:');
        console.warn('   1. 完全退出Cursor应用');
        console.warn('   2. 在"系统偏好设置" → "安全性与隐私" → "隐私" → "完整磁盘访问"中添加此应用');
        console.warn('   3. 或者在终端中运行: sudo chmod -R 644 ~/Library/Application\\ Support/Cursor/machine*');
      }
      
      if (otherErrors.length > 0) {
        console.warn('\n⚠️ 其他错误的文件:');
        otherErrors.forEach(ef => {
          console.warn(`   ${ef.path}: ${ef.error} (${ef.code || '无错误代码'})`);
        });
      }
      
      // 如果主要是权限问题，返回特定的错误信息
      if (permissionErrors.length > 0 && processedFiles === 0) {
        throw new Error(`权限不足：无法修改设备ID文件。请检查应用权限或以管理员身份运行。`);
      }
    }
    
    return true;
  } catch (error) {
    console.error(`重置机器ID失败: ${error.message}`);
    return false;
  }
}

// 获取用户数据路径 (已移除重复定义，使用上面更完整的版本)

// 添加验证卡密的处理器（用于校验，不更新数据库）
ipcMain.handle('verify-card-only', async (event, cardCode) => {
  try {
    console.log(`验证卡密（仅校验）: ${cardCode}`);

    // 使用新的后端验证接口
    const config = require('./config');
    const apiUrl = `${config.API_BASE_URL}/api/v1/keys/verify`;
    
    // 为测试目的，可以直接返回模拟数据
    // 实际使用时替换为真实API请求
    if (process.env.NODE_ENV === 'development') {
      console.log('开发模式: 返回模拟的卡密验证数据');
      
      const now = new Date();
      const endDate = new Date(now.getTime() + 31 * 24 * 60 * 60 * 1000);
      
      // 模拟后端API返回的数据格式
      const mockApiResponse = {
        success: true,
        message: "验证成功",
        cardCode: cardCode,
        cardType: "专用",
        startTime: now.toISOString().replace('T', ' ').substring(0, 19),
        endTime: endDate.toISOString().replace('T', ' ').substring(0, 19),
        useDays: 31,
        userId: `user_${Date.now()}`,
        email: `user_${Date.now()}@cursorrenew.com`,
        token: `token_${Date.now()}_${Math.random().toString(36).substring(2)}`
      };
      
      // 将模拟的API响应转换为前端期望的格式
      const cardInfo = {
        card: mockApiResponse.cardCode,
        address: mockApiResponse.cardType,
        start: mockApiResponse.startTime,
        end: mockApiResponse.endTime,
        usetime: mockApiResponse.useDays.toString()
      };
      
      const accountInfo = {
        userid: mockApiResponse.userId,
        email: mockApiResponse.email,
        token: mockApiResponse.token,
        current_time: now.toISOString().replace('T', ' ').substring(0, 19)
      };
      
      const result = {
        success: true,
        data: {
          card_info: cardInfo,
          account_info: accountInfo,
          remaining_days: 31
        }
      };

      // 保存到本地缓存
      saveCardToCache(cardCode, result.data);

      return result;
    }
    
    // 发送API请求到新后端验证接口
    try {
      // 获取机器码
      const machineId = generateMachineId();
      
      const response = await axios.post(apiUrl, { 
        key: cardCode,
        machineId: machineId
      });
      
      // 检查API响应
      if (response.status !== 200) {
        console.error(`❌ API请求失败: HTTP ${response.status}`);
        return {
          success: false,
          error: `API请求失败: HTTP状态码 ${response.status}`
        };
      }
      
      const data = response.data;
      
      // 检查Key是否有效
      if (!data.valid || !data.keyInfo) {
        console.error('❌ Key验证失败:', data.reason);
        let errorMsg = 'Key验证失败';
        
        if (data.reason === 'KEY_NOT_FOUND') {
          errorMsg = 'Key不存在，请检查是否输入正确';
        } else if (data.reason === 'KEY_NOT_ACTIVE') {
          errorMsg = 'Key已被停用，请联系客服';
        } else if (data.reason === 'NO_REMAINING_USES') {
          errorMsg = 'Key次数已用完，请购买新的Key';
        } else if (data.reason === 'KEY_NOT_YET_VALID') {
          errorMsg = 'Key尚未生效';
        } else if (data.reason === 'KEY_EXPIRED') {
          errorMsg = 'Key已过期';
        } else {
          errorMsg = data.reason || 'Key验证失败';
        }
        
        return {
          success: false,
          error: errorMsg
        };
      }
      
      const keyInfo = data.keyInfo;
      console.log('✅ Key验证成功:', keyInfo.displayText);
      
      // 构造前端期望的数据格式
      const now = new Date();
      let remainingDays = keyInfo.type === 'usage' ? keyInfo.remainingUses : keyInfo.remainingDays;
      
      const cardInfo = {
        card: cardCode,
        address: keyInfo.type === 'usage' ? '次数卡' : '期限卡',
        start: keyInfo.validFrom ? new Date(keyInfo.validFrom).toISOString().replace('T', ' ').substring(0, 19) : now.toISOString().replace('T', ' ').substring(0, 19),
        end: keyInfo.validUntil ? new Date(keyInfo.validUntil).toISOString().replace('T', ' ').substring(0, 19) : '',
        usetime: keyInfo.type === 'usage' ? `${keyInfo.remainingUses}` : `${keyInfo.remainingDays}`
      };
      
      const accountInfo = {
        userid: `user_${Date.now()}`,
        email: `key_${cardCode.substring(0, 8)}@cursorrenew.com`,
        token: '', // 验证时不返回token，只在续杯时返回
        current_time: now.toISOString().replace('T', ' ').substring(0, 19)
      };
      
      // 返回处理后的数据
      const result = {
        success: true,
        data: {
          card_info: cardInfo,
          account_info: accountInfo,
          remaining_days: remainingDays
        }
      };

      // 保存到本地缓存
      saveCardToCache(cardCode, result.data);

      return result;
    } catch (apiError) {
      console.error('API请求失败:', apiError);
      return {
        success: false,
        error: `API请求失败: ${apiError.message}`
      };
    }
  } catch (error) {
    console.error('验证卡密时出错:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// 检查缓存的卡密
ipcMain.handle('check-cached-card', async () => {
  try {
    console.log('检查本地缓存的卡密...');

    const cacheData = loadCardFromCache();
    if (!cacheData) {
      console.log('没有找到缓存的卡密');
      return { success: false, message: '没有缓存的卡密' };
    }

    // 检查是否过期
    if (isCachedCardExpired(cacheData)) {
      console.log('缓存的卡密已过期，清除缓存');
      clearCardCache();
      return { success: false, message: '缓存的卡密已过期' };
    }

    console.log('找到有效的缓存卡密:', cacheData.cardCode);
    return {
      success: true,
      data: {
        cardCode: cacheData.cardCode,
        cardData: cacheData.cardData,
        cachedAt: cacheData.cachedAt
      }
    };
  } catch (error) {
    console.error('检查缓存卡密失败:', error);
    return { success: false, error: error.message };
  }
});

// 清除缓存的卡密
ipcMain.handle('clear-cached-card', async () => {
  try {
    console.log('清除缓存的卡密...');
    const result = clearCardCache();
    return { success: result };
  } catch (error) {
    console.error('清除缓存卡密失败:', error);
    return { success: false, error: error.message };
  }
});

// 生成设备ID
function generateMachineId() {
  try {
    const { machineIdSync } = require('node-machine-id');
    return machineIdSync();
  } catch (error) {
    console.warn('无法获取真实的机器ID，使用随机ID:', error.message);
    const { v4: uuidv4 } = require('uuid');
    return uuidv4();
  }
}

// 卡密缓存文件路径
function getCardCacheFilePath() {
  const path = require('path');
  const os = require('os');
  const appDataPath = path.join(os.homedir(), '.cursor-renewal-cache');

  // 确保目录存在
  const fs = require('fs');
  if (!fs.existsSync(appDataPath)) {
    fs.mkdirSync(appDataPath, { recursive: true });
  }

  return path.join(appDataPath, 'card-cache.json');
}

// 保存卡密到本地缓存
function saveCardToCache(cardCode, cardData) {
  try {
    const fs = require('fs');
    const cacheFilePath = getCardCacheFilePath();

    // 创建安全的缓存数据，移除敏感的token信息
    const safeCardData = {
      card_info: cardData.card_info ? {
        card: cardData.card_info.card,
        address: cardData.card_info.address,
        start: cardData.card_info.start,
        end: cardData.card_info.end,
        usetime: cardData.card_info.usetime
      } : null,
      account_info: cardData.account_info ? {
        userid: cardData.account_info.userid,
        email: cardData.account_info.email,
        current_time: cardData.account_info.current_time
        // 注意：不保存token和refresh_token等敏感信息
      } : null,
      remaining_days: cardData.remaining_days
    };

    const cacheData = {
      cardCode: cardCode,
      cardData: safeCardData,
      cachedAt: new Date().toISOString(),
      expiresAt: cardData.card_info ? cardData.card_info.end : null
    };

    fs.writeFileSync(cacheFilePath, JSON.stringify(cacheData, null, 2), 'utf8');
    console.log('卡密已保存到本地缓存（已移除敏感信息）:', cacheFilePath);
    return true;
  } catch (error) {
    console.error('保存卡密缓存失败:', error);
    return false;
  }
}

// 从本地缓存读取卡密
function loadCardFromCache() {
  try {
    const fs = require('fs');
    const cacheFilePath = getCardCacheFilePath();

    if (!fs.existsSync(cacheFilePath)) {
      console.log('卡密缓存文件不存在');
      return null;
    }

    const cacheContent = fs.readFileSync(cacheFilePath, 'utf8');
    const cacheData = JSON.parse(cacheContent);

    console.log('从缓存加载卡密:', cacheData.cardCode);
    return cacheData;
  } catch (error) {
    console.error('读取卡密缓存失败:', error);
    return null;
  }
}

// 检查缓存的卡密是否过期
function isCachedCardExpired(cacheData) {
  if (!cacheData || !cacheData.expiresAt) {
    return true;
  }

  try {
    const expiresAt = new Date(cacheData.expiresAt);
    const now = new Date();

    const isExpired = now >= expiresAt;
    console.log(`卡密过期检查: 当前时间=${now.toISOString()}, 过期时间=${expiresAt.toISOString()}, 是否过期=${isExpired}`);

    return isExpired;
  } catch (error) {
    console.error('检查卡密过期时间失败:', error);
    return true;
  }
}

// 清除卡密缓存
function clearCardCache() {
  try {
    const fs = require('fs');
    const cacheFilePath = getCardCacheFilePath();

    if (fs.existsSync(cacheFilePath)) {
      fs.unlinkSync(cacheFilePath);
      console.log('卡密缓存已清除');
    }
    return true;
  } catch (error) {
    console.error('清除卡密缓存失败:', error);
    return false;
  }
}



// 添加续杯卡密的处理器（会更新数据库）
ipcMain.handle('verify-card', async (event, cardCode) => {
  try {
    console.log(`续杯卡密: ${cardCode}`);

    // 自动获取设备ID（使用系统真实机器码，不允许用户输入）
    const machineId = generateMachineId();
    console.log('📱 机器码（自动获取）:', machineId);

    // 构建API请求参数 - 使用新的代理后端
    const config = require('./config');
    const apiUrl = `${config.API_BASE_URL}/api/v1/token/renew`; // 使用新的代理后端
    const params = {
      key: cardCode,  // 改为 key 参数
      machineId: machineId  // 改为 machineId 参数
    };

    // 为测试目的，可以直接返回模拟数据
    if (process.env.NODE_ENV === 'development') {
      console.log('开发模式: 返回模拟的续杯数据');

      const now = new Date();
      const endDate = new Date(now.getTime() + 31 * 24 * 60 * 60 * 1000);

      // 模拟后端API返回的数据格式
      const mockApiResponse = {
        success: true,
        message: "续杯成功",
        cardCode: cardCode,
        cardType: "专用",
        startTime: now.toISOString().replace('T', ' ').substring(0, 19),
        endTime: endDate.toISOString().replace('T', ' ').substring(0, 19),
        useDays: 31,
        userId: `user_${Date.now()}`,
        email: `user_${Date.now()}@cursorrenew.com`,
        token: `token_${Date.now()}_${Math.random().toString(36).substring(2)}`
      };

      // 将模拟的API响应转换为前端期望的格式
      const cardInfo = {
        card: mockApiResponse.cardCode,
        address: mockApiResponse.cardType,
        start: mockApiResponse.startTime,
        end: mockApiResponse.endTime,
        usetime: mockApiResponse.useDays.toString()
      };

      const accountInfo = {
        userid: mockApiResponse.userId,
        email: mockApiResponse.email,
        token: mockApiResponse.token,
        current_time: now.toISOString().replace('T', ' ').substring(0, 19)
      };

      return {
        success: true,
        data: {
          card_info: cardInfo,
          account_info: accountInfo,
          remaining_days: 31
        }
      };
    }

    // 发送API请求
    try {
      const response = await axios.post(apiUrl, {
        key: cardCode,
        machineId: machineId
      });

      // 检查API响应
      if (response.status !== 200) {
        console.error(`❌ API请求失败: HTTP ${response.status}`);
        return {
          success: false,
          error: `API请求失败: HTTP状态码 ${response.status}`
        };
      }

      const data = response.data;

      // 检查是否有错误
      if (data.error) {
        console.error('❌ 后端返回错误:', data.error);
        
        // 根据错误代码返回友好提示
        let errorMsg = '续杯失败';
        const errorCode = data.error.code;
        const errorMessage = data.error.message || '';
        
        if (errorCode === 'INVALID_KEY' || errorMessage.includes('Invalid key')) {
          errorMsg = '卡密无效或已过期，请检查卡密是否正确';
        } else if (errorCode === 'NO_AVAILABLE_BINDING') {
          errorMsg = '暂无可用授权，请联系管理员';
        } else if (errorMessage.includes('Real backend')) {
          errorMsg = '系统授权失败，请联系客服';
        } else if (errorMessage.includes('remainingUses')) {
          errorMsg = '卡密次数已用完，请续费';
        } else {
          errorMsg = errorMessage || '续杯失败，请稍后重试';
        }
        
        return {
          success: false,
          error: errorMsg
        };
      }

      // 新后端API返回格式: { token, expiresAt, keyInfo }
      if (!data.token) {
        console.error('❌ 后端未返回有效token');
        return {
          success: false,
          error: '服务器未返回有效token，请联系管理员'
        };
      }

      console.log('✅ 成功获取token，有效期至:', data.expiresAt);
      console.log('📋 卡密信息:', data.keyInfo);

      // 使用后端返回的卡密信息
      const keyInfo = data.keyInfo || {};
      const now = new Date();
      
      // 根据卡密类型显示剩余信息
      let remainingDays = keyInfo.remainingDays || 0;
      let displayInfo = '';
      
      if (keyInfo.type === 'usage') {
        // 次数卡：显示剩余次数
        displayInfo = `剩余 ${keyInfo.remainingUses || 0} 次`;
        remainingDays = keyInfo.remainingUses || 0;
      } else {
        // 期限卡：显示剩余天数
        displayInfo = `剩余 ${remainingDays} 天`;
      }
      
      console.log(`💳 卡密状态: ${displayInfo}`);
      
      // 适配前端期望的数据格式
      const cardInfo = {
        card: cardCode || '',
        address: keyInfo.type === 'usage' ? '次数卡' : '期限卡',
        start: now.toISOString().replace('T', ' ').substring(0, 19),
        end: keyInfo.validUntil ? new Date(keyInfo.validUntil).toISOString().replace('T', ' ').substring(0, 19) : '',
        usetime: keyInfo.type === 'usage' ? `${keyInfo.remainingUses}次` : `${remainingDays}天`
      };

      const accountInfo = {
        userid: `user_${Date.now()}`,
        email: `user_${Date.now()}@cursorrenew.com`,
        token: data.token,
        current_time: now.toISOString().replace('T', ' ').substring(0, 19)
      };

      return {
        success: true,  // token获取成功
        data: {
          card_info: cardInfo,
          account_info: accountInfo,
          remaining_days: remainingDays
        }
      };
    } catch (apiError) {
      console.error('API请求失败:', apiError);
      
      // 处理HTTP错误状态码
      if (apiError.response) {
        const status = apiError.response.status;
        const errorData = apiError.response.data;
        const errorMsg = errorData?.error?.message || '';
        const errorCode = errorData?.error?.code || '';
        
        console.log(`HTTP状态码: ${status}, 错误码: ${errorCode}, 错误消息: ${errorMsg}`);
        
        // 403 - 机器码已封禁
        if (status === 403 && errorCode === 'MACHINE_BANNED') {
          return {
            success: false,
            error: `⛔ 机器码已被封禁\n原因: ${errorData.error.banReason || '违规使用'}\n请联系管理员`
          };
        }
        
        // 429 - 冷却时间限制
        if (status === 429 && errorCode === 'RATE_LIMIT_COOLDOWN') {
          const remainingMinutes = errorData.error.remainingMinutes || 0;
          const nextAvailableAt = errorData.error.nextAvailableAt;
          
          let timeStr = '';
          if (nextAvailableAt) {
            try {
              const nextTime = new Date(nextAvailableAt);
              timeStr = nextTime.toLocaleString('zh-CN', {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
              });
            } catch (e) {
              timeStr = '';
            }
          }
          
          let errorMessage = `⚠️ 风控系统提示您，下次换号时间需要等待 ${remainingMinutes} 分钟`;
          if (timeStr) {
            errorMessage += `，为 ${timeStr}`;
          }
          
          return {
            success: false,
            error: errorMessage
          };
        }
        
        // 429 - 使用量超限（旧版本错误码）
        if (status === 429 || errorCode === 'USAGE_LIMIT_EXCEEDED') {
          return {
            success: false,
            error: '今日使用量已超量，请联系管理员'
          };
        }
        
        // 500 - 服务器内部错误（可能是使用量超限）
        if (status === 500) {
          if (errorMsg.includes('超过') || errorMsg.includes('超限') || errorMsg.includes('封禁')) {
            return {
              success: false,
              error: '今日使用量已超量，请联系管理员'
            };
          }
          return {
            success: false,
            error: '服务器内部错误，请联系管理员'
          };
        }
        
        // 502 - 代理服务器访问真实服务器失败
        if (status === 502) {
          return {
            success: false,
            error: '号码池出现问题，请联系管理员'
          };
        }
        
        // 503 - 服务不可用（无可用账号）
        if (status === 503) {
          const waitMinutes = errorData?.error?.waitMinutes || 0;
          if (waitMinutes > 0) {
            return {
              success: false,
              error: `号码池目前无可用账号\n需要等待约 ${waitMinutes} 分钟`
            };
          }
          return {
            success: false,
            error: '号码池目前无可用账号，请稍后再试'
          };
        }
        
        // 其他错误
        if (errorMsg) {
          return {
            success: false,
            error: errorMsg
          };
        }
      }
      
      return {
        success: false,
        error: `API请求失败: ${apiError.message}`
      };
    }
  } catch (error) {
    console.error('续杯卡密时出错:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// 记录卡密使用情况
ipcMain.handle('record-usage', async (event, cardCode) => {
  try {
    console.log(`记录卡密使用: ${cardCode}`);

    // 这里可以添加使用记录逻辑，比如：
    // 1. 记录到本地文件
    // 2. 发送到服务器
    // 3. 更新使用次数等

    // 目前只是简单记录日志
    const timestamp = new Date().toISOString();
    const logEntry = {
      cardCode: cardCode,
      timestamp: timestamp,
      action: 'renewal_completed'
    };

    console.log('使用记录:', logEntry);

    return { success: true, message: '使用记录已保存' };
  } catch (error) {
    console.error('记录使用失败:', error);
    return { success: false, error: error.message };
  }
});

// 添加查询卡密信息的处理器
ipcMain.handle('get-card-info', async (event, cardCode) => {
  try {
    console.log(`查询卡密信息: ${cardCode}`);

    // 获取设备ID
    const machineId = generateMachineId();
    
    // 构建API请求参数
    const apiUrl = `http://129.204.108.166:2486/csk/card/info/${cardCode}`; // 使用本地SpringBoot服务API地址
    
    // 为测试目的，可以直接返回模拟数据
    if (process.env.NODE_ENV === 'development') {
      console.log('开发模式: 返回模拟的卡密信息数据');
      
      const now = new Date();
      const endDate = new Date(now.getTime() + 31 * 24 * 60 * 60 * 1000);
      
      // 模拟查询成功的响应
      return {
        success: true,
        cardCode: cardCode,
        cardType: '专用',
        startTime: now.toISOString().replace('T', ' ').substring(0, 19),
        endTime: endDate.toISOString().replace('T', ' ').substring(0, 19),
        useDays: 31
      };
    }
    
    // 发送API请求
    try {
      const response = await axios.get(apiUrl);
      
      // 检查API响应
      if (response.status !== 200) {
        return {
          success: false,
          error: `API请求失败: HTTP状态码 ${response.status}`
        };
      }
      
      const data = response.data;
      
      // 将SpringBoot后端返回的扁平数据结构转换为前端期望的格式
      return {
        success: data.success,
        cardCode: data.cardCode || '',
        cardType: data.cardType || '专用',
        startTime: data.startTime || '',
        endTime: data.endTime || '',
        useDays: data.useDays || 31
      };
    } catch (apiError) {
      console.error('API请求失败:', apiError);
      return {
        success: false,
        error: `API请求失败: ${apiError.message}`
      };
    }
  } catch (error) {
    console.error('查询卡密信息时出错:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// 打开外部链接
ipcMain.handle('open-external-url', async (event, url) => {
  try {
    const { shell } = require('electron');
    await shell.openExternal(url);
    console.log(`已打开外部链接: ${url}`);
    return { success: true };
  } catch (error) {
    console.error('打开外部链接失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

// 设备ID保护监控管理
ipcMain.handle('start-device-protection', async (event, storagePath, deviceIds) => {
  try {
    const result = startDeviceIdProtection(storagePath, deviceIds);
    return { success: result };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

ipcMain.handle('stop-device-protection', async (event) => {
  try {
    const result = stopDeviceIdProtection();
    return { success: result };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

ipcMain.handle('get-protection-status', async (event) => {
  return {
    success: true,
    isActive: isWatcherActive,
    monitoringFile: storageJsonPath,
    protectedIds: protectedDeviceIds ? Object.keys(protectedDeviceIds) : []
  };
});

// 窗口控制事件
ipcMain.on('minimize-window', () => {
  if (mainWindow) {
    mainWindow.minimize();
  }
});

ipcMain.on('close-window', () => {
  if (mainWindow) {
    mainWindow.close();
  }
});

// 借鉴Python版本的SQLite数据库更新函数 - 完整实现 (reset_machine_manual.py 第587-620行)
async function updateSQLiteDatabase(cursorDir, newIds) {
  try {
    console.log('🗄️ 开始更新SQLite数据库中的机器ID - 借鉴Python版本完整实现...');
    
    // 确定SQLite数据库路径
    const sqlitePath = path.join(cursorDir, 'User', 'globalStorage', 'state.vscdb');
    console.log(`📂 SQLite数据库路径: ${sqlitePath}`);
    
    if (!fs.existsSync(sqlitePath)) {
      console.log('📄 SQLite数据库文件不存在，跳过数据库更新');
      return { success: true, message: 'SQLite数据库文件不存在，跳过更新' };
    }
    
    // 创建数据库备份 - 借鉴Python版本的备份策略
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace(/T/, '_').split('.')[0];
    const backupPath = `${sqlitePath}.bak.${timestamp}`;
    try {
      fs.copyFileSync(sqlitePath, backupPath);
      console.log(`💾 已创建SQLite数据库备份: ${path.basename(backupPath)}`);
    } catch (backupError) {
      console.warn('⚠️ 创建SQLite备份失败:', backupError.message);
      // 不中断流程，继续执行
    }
    
    // 使用现有的CursorDatabase类
    const db = new CursorDatabase(sqlitePath);
    const connectResult = db.connect();
    
    if (!connectResult.success) {
      console.error('❌ 连接SQLite数据库失败:', connectResult.error);
      return connectResult;
    }
    
    console.log('🔗 已成功连接到SQLite数据库');
    
    // 检查表是否存在 - 借鉴Python版本的表检查逻辑
    const tableCheck = db.checkTable('ItemTable');
    if (!tableCheck.success || (!tableCheck.exists && !tableCheck.simulation)) {
      console.log('📋 ItemTable表不存在，尝试创建新表...');
      
      // 如果不是模拟模式且表不存在，尝试创建表
      if (!tableCheck.simulation) {
        try {
          // 这里借鉴Python版本的表创建逻辑 (reset_machine_manual.py 第595-600行)
          db.createTable('ItemTable', {
            key: 'TEXT PRIMARY KEY',
            value: 'TEXT'
          });
          console.log('✅ 已创建ItemTable表');
        } catch (createError) {
          console.warn('⚠️ 创建表失败，但继续尝试更新:', createError.message);
        }
      }
    }
    
    // 准备要更新的键值对 - 使用传入的完整ID映射
    let dbUpdates = {};
    
    if (newIds.completeIdMap) {
      // 如果传入了完整的ID映射，直接使用
      dbUpdates = newIds.completeIdMap;
    } else {
      // 否则构建标准的字段映射 - 借鉴Python版本的字段映射
      dbUpdates = {
        'telemetry.devDeviceId': newIds.devDeviceId || newIds['telemetry.devDeviceId'],
        'telemetry.machineId': newIds.machineId || newIds['telemetry.machineId'],
        'telemetry.macMachineId': newIds.macMachineId || newIds['telemetry.macMachineId'],
        'telemetry.sqmId': newIds.sqmId || newIds['telemetry.sqmId'],
        'storage.serviceMachineId': newIds.serviceMachineId || newIds['storage.serviceMachineId']
      };
    }
    
    console.log(`🔄 正在批量更新SQLite数据库字段 (共${Object.keys(dbUpdates).length}个)...`);
    let updatedCount = 0;
    let failedCount = 0;
    
    // 借鉴Python版本的批量更新逻辑 (reset_machine_manual.py 第602-611行)
    for (const [key, value] of Object.entries(dbUpdates)) {
      if (!value) {
        console.warn(`⚠️ 跳过空值字段: ${key}`);
        continue;
      }
      
      try {
        const updateResult = db.updateItem(key, value);
        if (updateResult.success || updateResult.simulation) {
          const displayValue = typeof value === 'string' && value.length > 20 
            ? value.substring(0, 12) + '...' 
            : value;
          console.log(`  ✅ 已更新 ${key}: ${displayValue}`);
          updatedCount++;
        } else {
          console.warn(`  ❌ 更新 ${key} 失败: ${updateResult.error}`);
          failedCount++;
        }
      } catch (updateError) {
        console.warn(`  ❌ 更新 ${key} 时出错: ${updateError.message}`);
        failedCount++;
      }
    }
    
    db.close();
    
    const totalFields = Object.keys(dbUpdates).length;
    const successMessage = `SQLite数据库更新完成: ${updatedCount}/${totalFields} 成功`;
    
    if (failedCount > 0) {
      console.log(`🗄️ ${successMessage}, ${failedCount} 失败`);
    } else {
      console.log(`🗄️ ${successMessage}`);
    }
    
    return {
      success: updatedCount > 0, // 只要有字段更新成功就认为成功
      message: successMessage,
      updatedCount,
      failedCount,
      totalFields,
      // 返回更新的字段列表，供其他函数使用
      updatedFields: Object.keys(dbUpdates).filter(key => dbUpdates[key])
    };
    
  } catch (error) {
    console.error('❌ 更新SQLite数据库失败:', error);
    return {
      success: false,
      error: error.message,
      updatedCount: 0,
      failedCount: 0,
      totalFields: 0
    };
  }
}

// macOS系统级Platform UUID更新函数 - 借鉴Python版本实现 (reset_machine_manual.py 第699-713行)
async function updateMacOSPlatformUUID(newIds) {
  const { exec } = require('child_process');
  const { promisify } = require('util');
  const execAsync = promisify(exec);
  
  try {
    console.log('🔧 开始更新macOS Platform UUID - 借鉴Python版本实现...');
    
    // macOS Platform UUID配置文件路径 (Python版本中的路径)
    const uuidFile = "/var/root/Library/Preferences/SystemConfiguration/com.apple.platform.uuid.plist";
    
    // 检查文件是否存在
    console.log(`📂 检查Platform UUID文件: ${uuidFile}`);
    
    try {
      await execAsync(`test -f "${uuidFile}"`);
      console.log('✅ Platform UUID文件存在');
    } catch (checkError) {
      console.log('⚠️ Platform UUID文件不存在，跳过系统级UUID更新');
      return {
        success: true,
        message: 'Platform UUID文件不存在，跳过更新',
        skipped: true
      };
    }
    
    // 获取要使用的UUID - 使用macMachineId
    const newUUID = newIds['telemetry.macMachineId'] || newIds.macMachineId;
    if (!newUUID) {
      throw new Error('未找到macMachineId用于更新Platform UUID');
    }
    
    console.log(`🔄 正在使用macMachineId更新Platform UUID: ${newUUID.substring(0, 16)}...`);
    
    // 使用plutil命令更新plist文件 - 借鉴Python版本的命令
    // 注意：这个操作需要sudo权限
    const plUtilCommand = `sudo plutil -replace "UUID" -string "${newUUID}" "${uuidFile}"`;
    
    console.log('⚡ 执行plutil命令 (需要管理员权限)...');
    console.log(`📝 命令: ${plUtilCommand}`);
    
    try {
      const { stdout, stderr } = await execAsync(plUtilCommand);
      
      if (stderr && stderr.trim() !== '') {
        console.warn('⚠️ plutil命令执行时有警告:', stderr);
      }
      
      console.log('✅ plutil命令执行成功');
      
      // 验证更新结果
      try {
        const verifyCommand = `sudo plutil -p "${uuidFile}" | grep UUID`;
        const { stdout: verifyOutput } = await execAsync(verifyCommand);
        
        if (verifyOutput.includes(newUUID.substring(0, 8))) {
          console.log('✅ Platform UUID更新验证成功');
        } else {
          console.warn('⚠️ Platform UUID更新验证失败');
        }
      } catch (verifyError) {
        console.warn('⚠️ 无法验证Platform UUID更新结果:', verifyError.message);
      }
      
      return {
        success: true,
        message: 'macOS Platform UUID更新成功',
        updatedUUID: newUUID,
        command: plUtilCommand
      };
      
    } catch (execError) {
      // 解析具体的错误类型
      if (execError.message.includes('Permission denied') || 
          execError.message.includes('Operation not permitted')) {
        console.error('❌ 权限不足：无法更新Platform UUID');
        console.log('💡 解决方案:');
        console.log('   1. 确保以管理员身份运行应用');
        console.log('   2. 在"系统偏好设置" → "安全性与隐私" → "隐私" → "完整磁盘访问"中添加此应用');
        console.log('   3. 或者在终端中运行: sudo chmod 644 /var/root/Library/Preferences/SystemConfiguration/com.apple.platform.uuid.plist');
        
        return {
          success: false,
          error: '权限不足，无法更新macOS Platform UUID',
          permissionDenied: true,
          suggestions: [
            '以管理员身份运行应用',
            '在系统偏好设置中授予完整磁盘访问权限',
            '手动调整plist文件权限'
          ]
        };
      } else if (execError.message.includes('command not found')) {
        console.error('❌ plutil命令不存在');
        return {
          success: false,
          error: 'plutil命令不可用',
          missingCommand: true
        };
      } else {
        console.error('❌ plutil命令执行失败:', execError.message);
        throw execError;
      }
    }
    
  } catch (error) {
    console.error('❌ 更新macOS Platform UUID失败:', error.message);
    return {
      success: false,
      error: error.message,
      details: '系统级UUID更新失败，但不影响其他功能'
    };
  }
}

// 更新独立machineId文件 - 借鉴Python版本实现 (reset_machine_manual.py 第782-819行)
async function updateMachineIdFile(cursorDir, machineId) {
  try {
    console.log('📝 开始更新独立machineId文件 - 借鉴Python版本完整实现...');
    
    if (!machineId) {
      throw new Error('machineId参数为空');
    }
    
    // 定义各种可能的machineId文件路径 - 基于Python版本的路径映射
    const possiblePaths = [];
    
    if (process.platform === 'darwin') {
      // macOS特定的文件路径
      possiblePaths.push(
        path.join(cursorDir, 'machineId'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'machineId'),
        path.join(cursorDir, 'User', 'workspaceStorage', 'machine-id'),
        path.join(cursorDir, 'logs', 'machine-id')
      );
    } else if (process.platform === 'win32') {
      // Windows特定的文件路径
      possiblePaths.push(
        path.join(cursorDir, 'machineId'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'machineId')
      );
    } else {
      // Linux特定的文件路径
      possiblePaths.push(
        path.join(cursorDir, 'machineId'),
        path.join(cursorDir, 'machineid'),
        path.join(cursorDir, 'User', 'globalStorage', 'machine-id'),
        path.join(cursorDir, 'User', 'globalStorage', 'machineId')
      );
    }
    
    console.log(`🔍 检查 ${possiblePaths.length} 个可能的machineId文件路径...`);
    
    let processedFiles = 0;
    let createdFiles = 0;
    let updatedFiles = 0;
    let errorFiles = [];
    
    for (const filePath of possiblePaths) {
      try {
        // 确保目录存在
        const dir = path.dirname(filePath);
        if (!fs.existsSync(dir)) {
          fs.mkdirSync(dir, { recursive: true });
          console.log(`📂 已创建目录: ${dir}`);
        }
        
        const fileExists = fs.existsSync(filePath);
        
        // 如果文件存在，检查权限
        if (fileExists) {
          try {
            fs.accessSync(filePath, fs.constants.R_OK | fs.constants.W_OK);
          } catch (accessError) {
            console.warn(`⚠️ 文件权限不足: ${filePath}`);
            console.warn(`权限错误: ${accessError.message}`);
            
            // 提供权限修复建议
            if (accessError.code === 'EACCES') {
              console.warn('建议解决方案:');
              console.warn(`  chmod 644 "${filePath}"`);
              console.warn('  或者以管理员权限运行应用');
            }
            
            errorFiles.push({
              path: filePath,
              error: accessError.message,
              code: accessError.code,
              needsPermission: true
            });
            continue;
          }
        }
        
        // 创建备份（如果文件存在）- 借鉴Python版本的备份策略
        if (fileExists) {
          try {
            const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace(/T/, '_').split('.')[0];
            const backupPath = `${filePath}.backup.${timestamp}`;
            fs.copyFileSync(filePath, backupPath);
            console.log(`💾 已创建备份: ${path.basename(backupPath)}`);
          } catch (backupError) {
            console.warn(`⚠️ 创建备份失败: ${filePath}, ${backupError.message}`);
            // 不中断流程，继续执行
          }
        }
        
        // 写入新的machineId
        fs.writeFileSync(filePath, machineId, 'utf8');
        
        if (fileExists) {
          console.log(`✏️ 已更新: ${filePath}`);
          updatedFiles++;
        } else {
          console.log(`➕ 已创建: ${filePath}`);
          createdFiles++;
        }
        
        processedFiles++;
        
      } catch (fileError) {
        console.warn(`❌ 处理文件失败: ${filePath}`);
        console.warn(`错误详情: ${fileError.message}`);
        
        // 根据错误类型提供具体建议
        if (fileError.code === 'EACCES') {
          console.warn('❌ 权限被拒绝 - 需要管理员权限');
          console.warn('💡 解决方案:');
          console.warn('   1. 确保Cursor应用已完全退出');
          console.warn('   2. 在系统偏好设置中授予应用完整磁盘访问权限');
          console.warn(`   3. 手动修复权限: chmod 644 "${filePath}"`);
        } else if (fileError.code === 'ENOENT') {
          console.warn('📁 目录不存在 - 将尝试创建');
        } else if (fileError.code === 'EBUSY' || fileError.code === 'EMFILE') {
          console.warn('🔒 文件被占用 - Cursor可能仍在运行');
          console.warn('💡 请完全退出Cursor后重试');
        } else {
          console.warn(`❓ 未知错误: ${fileError.code || '无错误代码'}`);
        }
        
        errorFiles.push({
          path: filePath,
          error: fileError.message,
          code: fileError.code,
          needsPermission: fileError.code === 'EACCES'
        });
      }
    }
    
    // 报告处理结果
    console.log(`📊 machineId文件处理完成: 成功处理 ${processedFiles}/${possiblePaths.length} 个路径`);
    console.log(`   ✅ 更新了 ${updatedFiles} 个现有文件`);
    console.log(`   ➕ 创建了 ${createdFiles} 个新文件`);
    
    if (errorFiles.length > 0) {
      console.warn(`⚠️ 处理失败的文件数量: ${errorFiles.length}`);
      
      // 统计权限错误
      const permissionErrors = errorFiles.filter(ef => ef.needsPermission);
      const otherErrors = errorFiles.filter(ef => !ef.needsPermission);
      
      if (permissionErrors.length > 0) {
        console.warn('\n❌ 权限错误的文件:');
        permissionErrors.forEach(ef => {
          console.warn(`   ${ef.path}: ${ef.error}`);
        });
        console.warn('\n🔧 权限问题解决建议:');
        console.warn('   1. 完全退出Cursor应用');
        console.warn('   2. 在"系统偏好设置" → "安全性与隐私" → "隐私" → "完整磁盘访问"中添加此应用');
        if (process.platform === 'darwin') {
          console.warn('   3. 或者在终端中运行: sudo chmod -R 644 ~/Library/Application\\ Support/Cursor/machine*');
        }
      }
      
      if (otherErrors.length > 0) {
        console.warn('\n⚠️ 其他错误的文件:');
        otherErrors.forEach(ef => {
          console.warn(`   ${ef.path}: ${ef.error} (${ef.code || '无错误代码'})`);
        });
      }
    }
    
    // 判断整体成功状态
    const success = processedFiles > 0;
    
    if (!success && errorFiles.length > 0) {
      const permissionErrors = errorFiles.filter(ef => ef.needsPermission);
      if (permissionErrors.length > 0) {
        throw new Error(`权限不足：无法修改machineId文件。请检查应用权限或以管理员身份运行。`);
      }
    }
    
    return {
      success: success,
      message: success ? `machineId文件更新成功，处理了 ${processedFiles} 个文件` : '所有machineId文件处理失败',
      processedFiles: processedFiles,
      updatedFiles: updatedFiles,
      createdFiles: createdFiles,
      errorFiles: errorFiles.length,
      machineId: machineId
    };
    
  } catch (error) {
    console.error('❌ 更新独立machineId文件失败:', error.message);
    return {
      success: false,
      error: error.message,
      processedFiles: 0,
      updatedFiles: 0,
      createdFiles: 0,
      errorFiles: 0
    };
  }
}

// 修改workbench.desktop.main.js文件 - 借鉴Python版本实现 (reset_machine_manual.py 第325-397行)
async function modifyWorkbenchJS(pathsInfo) {
  const tempfile = require('temp');
  
  try {
    console.log('🎨 开始修改workbench.desktop.main.js - 借鉴Python版本完整实现...');
    
    // 构建workbench.js路径
    const workbenchPath = path.join(pathsInfo.basePath, 'out', 'vs', 'workbench', 'workbench.desktop.main.js');
    console.log(`📂 workbench.js路径: ${workbenchPath}`);
    
    if (!fs.existsSync(workbenchPath)) {
      throw new Error(`workbench.js文件不存在: ${workbenchPath}`);
    }
    
    // 保存原始文件权限
    const originalStat = fs.statSync(workbenchPath);
    const originalMode = originalStat.mode;
    const originalUid = originalStat.uid;
    const originalGid = originalStat.gid;
    
    // 创建备份
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace(/T/, '_').split('.')[0];
    const backupPath = `${workbenchPath}.backup.${timestamp}`;
    try {
      fs.copyFileSync(workbenchPath, backupPath);
      console.log(`💾 已创建workbench.js备份: ${path.basename(backupPath)}`);
    } catch (backupError) {
      console.warn('⚠️ 创建备份失败:', backupError.message);
    }
    
    // 读取文件内容
    const content = fs.readFileSync(workbenchPath, 'utf8');
    console.log(`📖 已读取workbench.js，文件大小: ${content.length} 字节`);
    
    // 借鉴Python版本的替换模式 (reset_machine_manual.py 第342-360行)
    const patterns = {
      // 通用按钮替换模式
      'B(k,D(Ln,{title:"Upgrade to Pro",size:"small",get codicon(){return A.rocket},get onClick(){return t.pay}}),null)': 
        'B(k,D(Ln,{title:"yeongpin GitHub",size:"small",get codicon(){return A.github},get onClick(){return function(){window.open("https://github.com/yeongpin/cursor-free-vip","_blank")}}}),null)',
      
      // Windows/Linux/Mac 通用按钮替换模式
      'M(x,I(as,{title:"Upgrade to Pro",size:"small",get codicon(){return $.rocket},get onClick(){return t.pay}}),null)':
        'M(x,I(as,{title:"yeongpin GitHub",size:"small",get codicon(){return $.rocket},get onClick(){return function(){window.open("https://github.com/yeongpin/cursor-free-vip","_blank")}}}),null)',
      
      // Badge 替换
      '<div>Pro Trial': '<div>Pro',
      
      // Auto-select替换
      'py-1">Auto-select': 'py-1">Bypass-Version-Pin',
      
      // Token限制绕过
      'async getEffectiveTokenLimit(e){const n=e.modelName;if(!n)return 2e5;':
        'async getEffectiveTokenLimit(e){return 9000000;const n=e.modelName;if(!n)return 9e5;',
      
      // Pro 状态显示
      'var DWr=ne("<div class=settings__item_description>You are currently signed in with <strong></strong>.");':
        'var DWr=ne("<div class=settings__item_description>You are currently signed in with <strong></strong>. <h1>Pro</h1>");',
      
      // 隐藏通知Toast
      'notifications-toasts': 'notifications-toasts hidden'
    };
    
    let newContent = content;
    let modifiedCount = 0;
    
    // 应用所有替换模式
    console.log('🔄 正在应用Python风格的替换模式...');
    for (const [oldPattern, newPattern] of Object.entries(patterns)) {
      if (newContent.includes(oldPattern)) {
        newContent = newContent.replace(new RegExp(escapeRegExp(oldPattern), 'g'), newPattern);
        modifiedCount++;
        console.log(`  ✅ 已应用模式: ${oldPattern.substring(0, 30)}...`);
      }
    }
    
    if (modifiedCount === 0) {
      console.log('📋 未找到匹配的模式，workbench.js可能已被修改或版本不兼容');
      return {
        success: true,
        message: '未找到需要修改的模式',
        modified: false,
        modifiedCount: 0
      };
    }
    
    // 创建临时文件
    const tempFile = tempfile.tmpNameSync({ suffix: '.js' });
    fs.writeFileSync(tempFile, newContent, 'utf8');
    
    // 原子性替换原文件
    if (fs.existsSync(workbenchPath)) {
      fs.unlinkSync(workbenchPath);
    }
    fs.copyFileSync(tempFile, workbenchPath);
    fs.unlinkSync(tempFile);
    
    // 恢复原始权限
    try {
      fs.chmodSync(workbenchPath, originalMode);
      if (process.platform !== 'win32') {
        fs.chownSync(workbenchPath, originalUid, originalGid);
      }
      console.log('🔒 已恢复文件权限');
    } catch (permError) {
      console.warn('⚠️ 恢复权限失败:', permError.message);
    }
    
    console.log(`✅ workbench.js修改完成，应用了 ${modifiedCount} 个替换模式`);
    
    return {
      success: true,
      message: `workbench.js修改成功，应用了 ${modifiedCount} 个模式`,
      modified: true,
      modifiedCount: modifiedCount,
      backupPath: backupPath
    };
    
  } catch (error) {
    console.error('❌ 修改workbench.js失败:', error.message);
    return {
      success: false,
      error: error.message,
      modified: false,
      modifiedCount: 0
    };
  }
}

// ============================================================================
// 辅助函数：CSP UUID 相关（借鉴 cursor-reset-code 插件）
// ============================================================================

/**
 * 辅助函数：生成UUID（兼容不同Node.js版本）
 */
function generateUUID() {
  // Node.js 15.6.0+ 支持 crypto.randomUUID()
  if (crypto.randomUUID) {
    return crypto.randomUUID();
  }
  
  // 兼容旧版本的UUID生成
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
}

/**
 * 检测文件中是否包含 CSP UUID 标记
 */
function detectCSPTokens(content) {
  // 格式: /*csp1*/"uuid"/*1csp*/ 和 /*csp4*/"uuid"/*4csp*/
  const csp1Regex = /\/\*csp1\*\/['"]([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})['"]\/\*1csp\*\//i;
  const csp4Regex = /\/\*csp4\*\/['"]([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})['"]\/\*4csp\*\//i;
  
  const csp1Match = content.match(csp1Regex);
  const csp4Match = content.match(csp4Regex);
  
  return {
    hasCsp1: !!csp1Match,
    hasCsp4: !!csp4Match,
    csp1Value: csp1Match ? csp1Match[1] : null,
    csp4Value: csp4Match ? csp4Match[1] : null
  };
}

/**
 * 替换 CSP UUID 标记（借鉴 cursor-reset-code 插件）
 */
function replaceCSPUUIDs(content) {
  const newCsp1UUID = generateUUID();
  const newCsp4UUID = generateUUID();
  
  let modified = false;
  let replacedCount = 0;
  
  // 替换 csp1 UUID - 格式: /*csp1*/"uuid"/*1csp*/
  const csp1Regex = /\/\*csp1\*\/['"]([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})['"]\/\*1csp\*\//gi;
  if (csp1Regex.test(content)) {
    const oldContent = content;
    content = content.replace(csp1Regex, `/*csp1*/"${newCsp1UUID}"/*1csp*/`);
    if (oldContent !== content) {
      modified = true;
      replacedCount++;
      console.log(`  ✅ 已替换 csp1 UUID: ${newCsp1UUID.substring(0, 8)}...`);
    }
  }
  
  // 替换 csp4 UUID - 格式: /*csp4*/"uuid"/*4csp*/
  const csp4Regex = /\/\*csp4\*\/['"]([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})['"]\/\*4csp\*\//gi;
  if (csp4Regex.test(content)) {
    const oldContent = content;
    content = content.replace(csp4Regex, `/*csp4*/"${newCsp4UUID}"/*4csp*/`);
    if (oldContent !== content) {
      modified = true;
      replacedCount++;
      console.log(`  ✅ 已替换 csp4 UUID: ${newCsp4UUID.substring(0, 8)}...`);
    }
  }
  
  return {
    content,
    modified,
    replacedCount
  };
}

// ============================================================================
// 修改main.js文件 - 增强版（原有功能 + CSP UUID替换）
// ============================================================================
async function modifyMainJS(pathsInfo) {
  const tempfile = require('temp');
  
  try {
    console.log('🔧 开始修改main.js - 增强版（原有功能 + CSP UUID替换）...');
    
    // 构建main.js路径
    const mainPath = path.join(pathsInfo.basePath, 'out', 'main.js');
    console.log(`📂 main.js路径: ${mainPath}`);
    
    if (!fs.existsSync(mainPath)) {
      throw new Error(`main.js文件不存在: ${mainPath}`);
    }
    
    // 保存原始文件权限
    const originalStat = fs.statSync(mainPath);
    const originalMode = originalStat.mode;
    const originalUid = originalStat.uid;
    const originalGid = originalStat.gid;
    
    // 创建备份
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').replace(/T/, '_').split('.')[0];
    const backupPath = `${mainPath}.backup.${timestamp}`;
    try {
      fs.copyFileSync(mainPath, backupPath);
      console.log(`💾 已创建main.js备份: ${path.basename(backupPath)}`);
    } catch (backupError) {
      console.warn('⚠️ 创建备份失败:', backupError.message);
    }
    
    // 读取文件内容
    let content = fs.readFileSync(mainPath, 'utf8');
    console.log(`📖 已读取main.js，文件大小: ${content.length} 字节`);
    
    // ========================================================================
    // 第一部分：原有的 getMachineId/getMacMachineId 函数修改
    // ========================================================================
    console.log('\n🔄 [步骤1] 应用原有的 getMachineId 修改模式...');
    
    const machineIdPatterns = [
      {
        // getMachineId函数模式
        regex: /async getMachineId\(\)\{return [^??]+\?\?([^}]+)\}/g,
        replacement: 'async getMachineId(){return $1}',
        description: 'getMachineId函数'
      },
      {
        // getMacMachineId函数模式
        regex: /async getMacMachineId\(\)\{return [^??]+\?\?([^}]+)\}/g,
        replacement: 'async getMacMachineId(){return $1}',
        description: 'getMacMachineId函数'
      }
    ];
    
    let machineIdModified = 0;
    for (const pattern of machineIdPatterns) {
      const testRegex = new RegExp(pattern.regex.source, pattern.regex.flags);
      if (testRegex.test(content)) {
        const beforeReplace = content;
        content = content.replace(pattern.regex, pattern.replacement);
        
        if (content !== beforeReplace) {
          machineIdModified++;
          console.log(`  ✅ 已应用模式: ${pattern.description}`);
        }
      }
    }
    
    if (machineIdModified === 0) {
      console.log('  📋 未找到 getMachineId 相关函数，可能已被修改或版本不兼容');
    } else {
      console.log(`  ✅ getMachineId 修改完成，应用了 ${machineIdModified} 个模式`);
    }
    
    // ========================================================================
    // 第二部分：新增的 CSP UUID 替换（借鉴 cursor-reset-code 插件）
    // ========================================================================
    console.log('\n🔄 [步骤2] 应用 CSP UUID 替换（借鉴插件功能）...');
    
    // 先检测是否存在 CSP 标记
    const cspDetection = detectCSPTokens(content);
    console.log(`📋 CSP标记检测结果:`);
    console.log(`  • csp1 标记: ${cspDetection.hasCsp1 ? '✅ 存在' : '❌ 不存在'}`);
    console.log(`  • csp4 标记: ${cspDetection.hasCsp4 ? '✅ 存在' : '❌ 不存在'}`);
    
    let cspModified = 0;
    if (cspDetection.hasCsp1 || cspDetection.hasCsp4) {
      if (cspDetection.hasCsp1) {
        console.log(`  📌 原 csp1 UUID: ${cspDetection.csp1Value}`);
      }
      if (cspDetection.hasCsp4) {
        console.log(`  📌 原 csp4 UUID: ${cspDetection.csp4Value}`);
      }
      
      const cspResult = replaceCSPUUIDs(content);
      content = cspResult.content;
      cspModified = cspResult.replacedCount;
      
      if (cspResult.modified) {
        console.log(`  ✅ CSP UUID 替换完成，更新了 ${cspModified} 个UUID`);
      }
    } else {
      console.log('  📋 未找到 CSP UUID 标记，跳过此步骤');
      console.log('  💡 这可能是较旧版本的Cursor，或main.js格式不同');
    }
    
    // ========================================================================
    // 总结修改结果
    // ========================================================================
    const totalModified = machineIdModified + cspModified;
    console.log(`\n📊 修改总结:`);
    console.log(`  • getMachineId 函数修改: ${machineIdModified} 处`);
    console.log(`  • CSP UUID 替换: ${cspModified} 处`);
    console.log(`  • 总计: ${totalModified} 处`);
    
    if (totalModified === 0) {
      console.log('⚠️ 未进行任何修改，main.js可能已被修改或版本不兼容');
      return {
        success: true,
        message: '未找到需要修改的内容',
        modified: false,
        modifiedCount: 0,
        details: {
          machineIdModified: 0,
          cspModified: 0
        }
      };
    }
    
    // ========================================================================
    // 写入文件
    // ========================================================================
    console.log('\n💾 正在保存修改...');
    
    // 创建临时文件
    const tempFile = tempfile.tmpNameSync({ suffix: '.js' });
    fs.writeFileSync(tempFile, content, 'utf8');
    
    // 原子性替换原文件
    if (fs.existsSync(mainPath)) {
      fs.unlinkSync(mainPath);
    }
    fs.copyFileSync(tempFile, mainPath);
    fs.unlinkSync(tempFile);
    
    // 恢复原始权限
    try {
      fs.chmodSync(mainPath, originalMode);
      if (process.platform !== 'win32') {
        fs.chownSync(mainPath, originalUid, originalGid);
      }
      console.log('🔒 已恢复文件权限');
    } catch (permError) {
      console.warn('⚠️ 恢复权限失败:', permError.message);
    }
    
    console.log(`\n✅ main.js增强修改完成！`);
    console.log(`📂 备份文件: ${backupPath}`);
    
    return {
      success: true,
      message: `main.js修改成功，应用了 ${totalModified} 个修改`,
      modified: true,
      modifiedCount: totalModified,
      backupPath: backupPath,
      details: {
        machineIdModified: machineIdModified,
        cspModified: cspModified
      }
    };
    
  } catch (error) {
    console.error('❌ 修改main.js失败:', error.message);
    return {
      success: false,
      error: error.message,
      modified: false,
      modifiedCount: 0
    };
  }
}

// 正则表达式转义函数
function escapeRegExp(string) {
  return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

// 检查应用版本更新
ipcMain.handle('check-app-version', async (event, currentVersion) => {
  try {
    console.log('🔍 检查应用版本更新...');
    console.log('📱 当前版本:', currentVersion);
    
    // 检测当前平台
    const platform = process.platform === 'darwin' ? 'mac' : 'windows';
    console.log('💻 当前平台:', platform);
    
    const config = require('./config');
    const apiUrl = `${config.API_BASE_URL}/api/v1/version/check?currentVersion=${currentVersion}&platform=${platform}`;
    
    console.log('📡 请求API:', apiUrl);
    
    const response = await axios.get(apiUrl);
    const result = response.data;
    
    console.log('📦 版本检查结果:', result);
    
    return result;
  } catch (error) {
    console.error('❌ 检查版本更新失败:', error);
    return {
      success: false,
      error: error.message
    };
  }
});

