const fs = require('fs');
const path = require('path');

let Database;

// 动态加载数据库模块
try {
  Database = require('better-sqlite3');
  console.log('better-sqlite3模块加载成功');
} catch (err) {
  console.error('better-sqlite3模块加载失败:', err.message);
  console.warn('数据库功能将使用模拟模式');
  Database = null;
}

class CursorDatabase {
  constructor(dbPath) {
    this.dbPath = dbPath;
    this.db = null;
    this.isSimulation = !Database;
  }

  // 连接数据库
  connect() {
    if (this.isSimulation) {
      console.log('数据库模拟模式：跳过连接');
      return { success: true, simulation: true };
    }

    try {
      if (!fs.existsSync(this.dbPath)) {
        return { success: false, error: '数据库文件不存在' };
      }

      this.db = new Database(this.dbPath);
      console.log('已成功连接到数据库');
      return { success: true };
    } catch (error) {
      console.error('连接数据库失败:', error);
      return { success: false, error: error.message };
    }
  }

  // 关闭数据库连接
  close() {
    if (this.db) {
      this.db.close();
      this.db = null;
      console.log('数据库连接已关闭');
    }
  }

  // 检查表是否存在
  checkTable(tableName = 'ItemTable') {
    if (this.isSimulation) {
      console.log(`数据库模拟模式：假设表 ${tableName} 存在`);
      return { success: true, exists: true };
    }

    try {
      const stmt = this.db.prepare("SELECT name FROM sqlite_master WHERE type='table' AND name=?");
      const result = stmt.get(tableName);
      const exists = !!result;
      
      console.log(`表 ${tableName} ${exists ? '存在' : '不存在'}`);
      return { success: true, exists };
    } catch (error) {
      console.error('检查表失败:', error);
      return { success: false, error: error.message };
    }
  }

  // 查询单个值
  getValue(key) {
    if (this.isSimulation) {
      console.log(`数据库模拟模式：查询键 ${key}`);
      return { success: true, value: null, simulation: true };
    }

    try {
      const stmt = this.db.prepare("SELECT value FROM ItemTable WHERE key = ?");
      const result = stmt.get(key);
      
      return { 
        success: true, 
        value: result ? result.value : null,
        found: !!result
      };
    } catch (error) {
      console.error(`查询键 ${key} 失败:`, error);
      return { success: false, error: error.message };
    }
  }

  // 更新或插入值
  setValue(key, value) {
    if (this.isSimulation) {
      console.log(`数据库模拟模式：设置键 ${key} = ${value}`);
      return { success: true, simulation: true };
    }

    try {
      // 检查键是否存在
      const checkStmt = this.db.prepare("SELECT COUNT(*) as count FROM ItemTable WHERE key = ?");
      const result = checkStmt.get(key);
      
      if (result.count > 0) {
        // 更新现有记录
        const updateStmt = this.db.prepare("UPDATE ItemTable SET value = ? WHERE key = ?");
        updateStmt.run(value, key);
        console.log(`更新键${key} -> ${value}`);
      } else {
        // 插入新记录
        const insertStmt = this.db.prepare("INSERT INTO ItemTable (key, value) VALUES (?, ?)");
        insertStmt.run(key, value);
        console.log(`插入键${key} -> ${value}`);
      }
      
      return { success: true, action: result.count > 0 ? 'updated' : 'inserted' };
    } catch (error) {
      console.error(`设置键 ${key} 失败:`, error);
      return { success: false, error: error.message };
    }
  }

  // 删除键
  deleteKey(key) {
    if (this.isSimulation) {
      console.log(`数据库模拟模式：删除键 ${key}`);
      return { success: true, simulation: true };
    }

    try {
      const stmt = this.db.prepare("DELETE FROM ItemTable WHERE key = ?");
      const result = stmt.run(key);
      
      console.log(`删除键 ${key}，影响行数: ${result.changes}`);
      return { success: true, changes: result.changes };
    } catch (error) {
      console.error(`删除键 ${key} 失败:`, error);
      return { success: false, error: error.message };
    }
  }

  // 批量更新（事务）
  batchUpdate(updates) {
    if (this.isSimulation) {
      console.log(`数据库模拟模式：批量更新 ${updates.length} 个键值对`);
      return { 
        success: true, 
        simulation: true,
        updatedKeys: updates.map(([key]) => key)
      };
    }

    try {
      const transaction = this.db.transaction((updates) => {
        const updatedKeys = [];
        
        for (const [key, value] of updates) {
          const result = this.setValue(key, value);
          if (result.success) {
            updatedKeys.push(key);
          } else {
            throw new Error(`设置键 ${key} 失败: ${result.error}`);
          }
        }
        
        return updatedKeys;
      });

      const updatedKeys = transaction(updates);
      console.log(`批量更新成功，更新了 ${updatedKeys.length} 个键`);
      
      return { success: true, updatedKeys };
    } catch (error) {
      console.error('批量更新失败:', error);
      return { success: false, error: error.message };
    }
  }

  // 更新Cursor认证信息
  updateCursorAuth(email, accessToken, refreshToken) {
    const updates = [];
    
    if (email) {
      updates.push(['cursorAuth/cachedEmail', email]);
    }
    if (accessToken) {
      updates.push(['cursorAuth/accessToken', accessToken]);
    }
    if (refreshToken) {
      updates.push(['cursorAuth/refreshToken', refreshToken]);
      updates.push(['cursorAuth/cachedSignUpType', 'Auth_0']);
    }

    // 首先删除serverConfig以激活新账号
    const deleteResult = this.deleteKey('cursorai/serverConfig');
    if (deleteResult.success) {
      console.log('成功删除cursorai/serverConfig，新账号已激活');
    }

    // 批量更新认证信息
    return this.batchUpdate(updates);
  }

  // 创建数据库备份
  createBackup() {
    if (this.isSimulation) {
      console.log('数据库模拟模式：跳过备份创建');
      return { success: true, simulation: true };
    }

    try {
      const backupPath = `${this.dbPath}.bak`;
      
      if (!fs.existsSync(backupPath)) {
        fs.copyFileSync(this.dbPath, backupPath);
        console.log(`已创建数据库备份: ${backupPath}`);
      } else {
        console.log('数据库备份已存在，跳过创建');
      }
      
      return { success: true, backupPath };
    } catch (error) {
      console.error('创建数据库备份失败:', error);
      return { success: false, error: error.message };
    }
  }
}

module.exports = CursorDatabase;
