/**
 * 后端负载均衡器 - Node.js版本
 * 支持多个Render后端实例的智能分流和故障切换
 * 使用axios进行HTTP请求
 */

const axios = require('axios');

class BackendLoadBalancer {
  constructor(backendUrls) {
    this.backends = backendUrls.map(url => ({
      url: url,
      healthy: true,
      failureCount: 0,
      lastCheck: Date.now(),
      responseTime: 0,
      requestCount: 0
    }));
    this.currentIndex = 0;
    this.maxFailures = 3; // 连续失败3次标记为不健康
    this.healthCheckInterval = 60000; // 60秒健康检查一次
    this.failureResetTime = 300000; // 5分钟后重置失败计数
    this.healthCheckTimer = null;
    this.maxRetriesPerBackend = 2; // 每个后端最多重试2次（总共3次尝试）
    
    // 启动健康检查
    this.startHealthCheck();
    
    console.log('🚀 负载均衡器已初始化，后端数量:', this.backends.length);
    this.backends.forEach((b, i) => {
      console.log(`  后端${i + 1}: ${b.url}`);
    });
  }

  /**
   * 获取下一个可用的后端（轮询策略）
   */
  getNextBackend() {
    // 优先从健康后端中选择
    const healthyBackends = this.backends.filter(b => b.healthy);
    
    if (healthyBackends.length === 0) {
      console.warn('⚠️ 所有后端都不健康，尝试轮询所有后端');
      // 如果所有后端都不健康，直接轮询所有后端（不重置状态）
      const backend = this.backends[this.currentIndex % this.backends.length];
      this.currentIndex++;
      return backend;
    }
    
    // 轮询选择健康后端
    const backend = healthyBackends[this.currentIndex % healthyBackends.length];
    this.currentIndex++;
    
    return backend;
  }

  /**
   * 获取响应时间最快的后端（最佳性能策略）
   */
  getFastestBackend() {
    const healthyBackends = this.backends.filter(b => b.healthy);
    
    if (healthyBackends.length === 0) {
      return this.backends[0];
    }
    
    // 按响应时间排序，选择最快的
    const sorted = [...healthyBackends].sort((a, b) => a.responseTime - b.responseTime);
    return sorted[0];
  }

  /**
   * 获取负载最小的后端（负载均衡策略）
   */
  getLeastLoadedBackend() {
    const healthyBackends = this.backends.filter(b => b.healthy);
    
    if (healthyBackends.length === 0) {
      return this.backends[0];
    }
    
    // 按请求数排序，选择请求最少的
    const sorted = [...healthyBackends].sort((a, b) => a.requestCount - b.requestCount);
    return sorted[0];
  }

  /**
   * 随机选择一个后端
   */
  getRandomBackend() {
    const healthyBackends = this.backends.filter(b => b.healthy);
    
    if (healthyBackends.length === 0) {
      return this.backends[0];
    }
    
    const index = Math.floor(Math.random() * healthyBackends.length);
    return healthyBackends[index];
  }

  /**
   * 标记后端请求成功
   */
  markSuccess(backend, responseTime) {
    backend.failureCount = 0;
    backend.healthy = true;
    backend.lastCheck = Date.now();
    backend.responseTime = responseTime;
    backend.requestCount++;
    
    console.log(`✅ 后端 ${backend.url} 请求成功 (${responseTime}ms)`);
  }

  /**
   * 标记后端请求失败
   */
  markFailure(backend, error) {
    backend.failureCount++;
    backend.lastCheck = Date.now();
    
    if (backend.failureCount >= this.maxFailures) {
      backend.healthy = false;
      console.error(`❌ 后端 ${backend.url} 标记为不健康 (连续失败${backend.failureCount}次)`);
    } else {
      console.warn(`⚠️ 后端 ${backend.url} 请求失败 (${backend.failureCount}/${this.maxFailures})`, error.message);
    }
    
    // 设置定时器：5分钟后重置失败计数
    setTimeout(() => {
      if (backend.failureCount > 0) {
        backend.failureCount = Math.max(0, backend.failureCount - 1);
        if (backend.failureCount < this.maxFailures) {
          backend.healthy = true;
          console.log(`🔄 后端 ${backend.url} 失败计数递减，重新标记为健康`);
        }
      }
    }, this.failureResetTime);
  }

  /**
   * 发送请求（带重试和故障转移）
   */
  async request(endpoint, options = {}) {
    // 准备后端列表：优先尝试健康后端，然后尝试不健康的后端
    const healthyBackends = this.backends.filter(b => b.healthy);
    const unhealthyBackends = this.backends.filter(b => !b.healthy);
    const backendQueue = [...healthyBackends, ...unhealthyBackends];
    
    if (backendQueue.length === 0) {
      throw new Error('没有配置任何后端');
    }
    
    let lastError;
    
    // 遍历所有后端
    for (let i = 0; i < backendQueue.length; i++) {
      const backend = backendQueue[i];
      const healthyStatus = backend.healthy ? '✅健康' : '⚠️不健康';
      
      // 对每个后端进行多次重试
      for (let retryCount = 0; retryCount <= this.maxRetriesPerBackend; retryCount++) {
        const url = `${backend.url}${endpoint}`;
        const startTime = Date.now();
        
        try {
          if (retryCount === 0) {
            console.log(`🚀 尝试后端 ${i + 1}/${backendQueue.length}: ${backend.url} [${healthyStatus}]`);
          } else {
            console.log(`🔄 重试后端 ${backend.url} (第${retryCount}次重试)`);
          }
          
          // 使用axios发送请求
          const response = await axios({
            url: url,
            method: options.method || 'GET',
            data: options.data,
            params: options.params,
            headers: {
              'Content-Type': 'application/json',
              ...options.headers
            },
            timeout: options.timeout || 30000,
            validateStatus: function (status) {
              return status >= 200 && status < 500; // 接受200-499的状态码
            }
          });
          
          const responseTime = Date.now() - startTime;
          
          // 检查HTTP状态码
          if (response.status < 200 || response.status >= 300) {
            throw new Error(`HTTP ${response.status}: ${JSON.stringify(response.data)}`);
          }
          
          // 标记成功
          this.markSuccess(backend, responseTime);
          
          return {
            success: true,
            data: response.data,
            backend: backend.url,
            responseTime: responseTime,
            status: response.status
          };
          
        } catch (error) {
          lastError = error;
          const responseTime = Date.now() - startTime;
          
          // 更详细的错误日志
          console.error(`❌ 后端请求失败:`, {
            backend: backend.url,
            retry: retryCount,
            error: error.message,
            status: error.response?.status,
            code: error.code
          });
          
          // 判断是否应该重试当前后端
          const isTimeout = error.code === 'ECONNABORTED' || error.code === 'ETIMEDOUT';
          const isNetworkError = !error.response;
          const shouldRetryBackend = (isTimeout || isNetworkError) && retryCount < this.maxRetriesPerBackend;
          
          if (shouldRetryBackend) {
            console.log(`⏳ 等待1秒后重试同一后端...`);
            await new Promise(resolve => setTimeout(resolve, 1000));
            continue; // 重试当前后端
          }
          
          // 标记失败（只在所有重试都失败后才标记）
          this.markFailure(backend, error);
          
          // 跳出重试循环，尝试下一个后端
          break;
        }
      }
      
      // 如果还有其他后端可用，继续尝试
      if (i < backendQueue.length - 1) {
        console.log(`🔄 切换到下一个后端...`);
        continue;
      }
    }
    
    // 所有后端都失败 - 提供更详细的错误信息
    const errorDetails = lastError?.response?.status 
      ? `HTTP ${lastError.response.status}` 
      : (lastError?.code || lastError?.message || '未知错误');
    throw new Error(`所有后端都不可用，最后一个错误: ${errorDetails}`);
  }

  /**
   * 健康检查
   */
  async healthCheck() {
    console.log('🏥 执行后端健康检查...');
    
    const checkPromises = this.backends.map(async (backend) => {
      try {
        const startTime = Date.now();
        
        const response = await axios.get(`${backend.url}/healthz`, {
          timeout: 5000,
          validateStatus: function (status) {
            return status === 200;
          }
        });
        
        const responseTime = Date.now() - startTime;
        
        backend.healthy = true;
        backend.failureCount = 0;
        backend.responseTime = responseTime;
        backend.lastCheck = Date.now();
        console.log(`✅ ${backend.url} 健康 (${responseTime}ms)`);
      } catch (error) {
        backend.failureCount++;
        if (backend.failureCount >= this.maxFailures) {
          backend.healthy = false;
        }
        backend.lastCheck = Date.now();
        console.error(`❌ ${backend.url} 不健康:`, error.message);
      }
    });
    
    // 并行检查所有后端
    await Promise.allSettled(checkPromises);
  }

  /**
   * 启动定期健康检查
   */
  startHealthCheck() {
    // 立即执行一次
    this.healthCheck().catch(err => {
      console.error('健康检查失败:', err);
    });
    
    // 定期执行
    this.healthCheckTimer = setInterval(() => {
      this.healthCheck().catch(err => {
        console.error('健康检查失败:', err);
      });
    }, this.healthCheckInterval);
  }

  /**
   * 停止健康检查
   */
  stopHealthCheck() {
    if (this.healthCheckTimer) {
      clearInterval(this.healthCheckTimer);
      this.healthCheckTimer = null;
      console.log('🛑 健康检查已停止');
    }
  }

  /**
   * 获取统计信息
   */
  getStats() {
    return this.backends.map(b => ({
      url: b.url,
      healthy: b.healthy,
      failureCount: b.failureCount,
      responseTime: b.responseTime,
      requestCount: b.requestCount,
      lastCheck: new Date(b.lastCheck).toISOString()
    }));
  }

  /**
   * 重置所有统计信息
   */
  resetStats() {
    this.backends.forEach(b => {
      b.requestCount = 0;
      b.responseTime = 0;
    });
  }

  /**
   * 销毁负载均衡器
   */
  destroy() {
    this.stopHealthCheck();
    console.log('🗑️ 负载均衡器已销毁');
  }
}

module.exports = BackendLoadBalancer;

